#include <vector>
#include <string>
#include "imgui.h"
#include <d3d11.h>
#include <d3dx11.h>

namespace notifications {
	enum class NotificationType {
		Info,
		Error,
		Success,
		Warning
	};

	struct notification {
		std::string message;
		float duration;
		NotificationType type; 
	};

	inline std::vector<notification> notifications;

	inline void add_notification(const std::string& message, float duration, NotificationType type = NotificationType::Info) {
		notifications.push_back({ message, duration, type });
	}

	inline void render_notifications(ID3D11ShaderResourceView* info, ID3D11ShaderResourceView* error, ID3D11ShaderResourceView* success, ID3D11ShaderResourceView* warning) {

		ImDrawList* draw_list = ImGui::GetBackgroundDrawList();

		float currentTime = ImGui::GetTime();

		ImVec2 window_pos = ImGui::GetWindowPos();
		ImVec2 window_size = ImGui::GetWindowSize();

		float max_notification_width = 0.0f;
		float total_notification_height = 0.0f;
		float notification_height = 48.0f; 

		for (int i = 0; i < notifications.size(); i++) {
			notification& n = notifications[i];

			if (n.duration <= 0.0f) {
				notifications.erase(notifications.begin() + i);
				i--;
				continue;
			}

			ImVec2 textSize = ImGui::CalcTextSize(n.message.c_str());
			float notification_width = textSize.x + 30.0f + 32.0f; 

			if (notification_width > max_notification_width) {
				max_notification_width = notification_width;
			}

			total_notification_height += notification_height; 
		}

		ImVec2 notifications_size(max_notification_width, total_notification_height);
		ImVec2 notifications_pos(window_pos.x + 15.0f, window_pos.y + window_size.y - 15.0f - notifications_size.y);

		for (int i = 0; i < notifications.size(); i++) {
			notification& n = notifications[i];

			if (n.duration <= 0.0f) {
				notifications.erase(notifications.begin() + i);
				i--;
				continue;
			}

			ImU32 bg_color, outline_color, text_color;
			ID3D11ShaderResourceView* icon = nullptr;

			switch (n.type) {
			case NotificationType::Info:
				bg_color = IM_COL32(239, 246, 255, 255); // #dbeafe
				outline_color = IM_COL32(96, 165, 250, 255); // #60a5fa
				text_color = IM_COL32(59, 130, 246, 255); // #3b82f6
				icon = info;
				break;
			case NotificationType::Error:
				bg_color = IM_COL32(254, 242, 242, 255); // #fee2e2
				outline_color = IM_COL32(248, 113, 113, 255); // #f87171
				text_color = IM_COL32(239, 68, 68, 255); // #ef4444
				icon = error;
				break;
			case NotificationType::Success:
				bg_color = IM_COL32(240, 253, 244, 255); // #dcfce7
				outline_color = IM_COL32(135, 239, 172, 255); // #86efac
				text_color = IM_COL32(34, 197, 94, 255); // #22c55e
				icon = success;
				break;
			case NotificationType::Warning:
				bg_color = IM_COL32(255, 251, 235, 255); // #fef3c7
				outline_color = IM_COL32(251, 191, 36, 255); // #fbbf24
				text_color = IM_COL32(245, 158, 11, 255); // #f59e0b
				icon = warning;
				break;
			}

			ImVec2 notification_pos = ImVec2(notifications_pos.x, notifications_pos.y + total_notification_height - notification_height);

			draw_list->AddRectFilled(notification_pos, ImVec2(notification_pos.x + notifications_size.x, notification_pos.y + notification_height), bg_color, 6.f);
			draw_list->AddRect(notification_pos, ImVec2(notification_pos.x + notifications_size.x, notification_pos.y + notification_height), outline_color, 6.f); 

			float icon_size = 19.0f;
			ImVec2 image_pos(notification_pos.x + 10.0f, notification_pos.y + (notification_height - icon_size) / 2.0f); 
			ImVec2 image_end(image_pos.x + icon_size, image_pos.y + icon_size); 
			draw_list->AddImage(icon, image_pos, image_end, ImVec2(0, 0), ImVec2(1, 1)); 

			ImVec2 text_pos(notification_pos.x + 13.0f + icon_size + 5.0f, 
				notification_pos.y + (notification_height - ImGui::CalcTextSize(n.message.c_str()).y) / 2.0f); 

			ImGui::PushFont(framework->geist_med);
			draw_list->AddText(text_pos, text_color, n.message.c_str());
			ImGui::PopFont();

		
			total_notification_height -= notification_height;
			notifications_pos.y -= 5.0f; 

			n.duration -= ImGui::GetIO().DeltaTime;
		}
	}
}


static unsigned char info_[1906] = {
	0x89, 0x50, 0x4E, 0x47, 0x0D, 0x0A, 0x1A, 0x0A, 0x00, 0x00, 0x00, 0x0D,
	0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x4B, 0x00, 0x00, 0x00, 0x4B,
	0x08, 0x06, 0x00, 0x00, 0x00, 0x38, 0x4E, 0x7A, 0xEA, 0x00, 0x00, 0x00,
	0x04, 0x73, 0x42, 0x49, 0x54, 0x08, 0x08, 0x08, 0x08, 0x7C, 0x08, 0x64,
	0x88, 0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x2E,
	0x23, 0x00, 0x00, 0x2E, 0x23, 0x01, 0x78, 0xA5, 0x3F, 0x76, 0x00, 0x00,
	0x00, 0x19, 0x74, 0x45, 0x58, 0x74, 0x53, 0x6F, 0x66, 0x74, 0x77, 0x61,
	0x72, 0x65, 0x00, 0x77, 0x77, 0x77, 0x2E, 0x69, 0x6E, 0x6B, 0x73, 0x63,
	0x61, 0x70, 0x65, 0x2E, 0x6F, 0x72, 0x67, 0x9B, 0xEE, 0x3C, 0x1A, 0x00,
	0x00, 0x06, 0xEF, 0x49, 0x44, 0x41, 0x54, 0x78, 0x9C, 0xED, 0x9C, 0x4F,
	0x6C, 0x1C, 0x57, 0x1D, 0xC7, 0x3F, 0xBF, 0xD9, 0x5D, 0x2F, 0x6D, 0x71,
	0x42, 0xE3, 0x5D, 0x12, 0x42, 0x4C, 0xEA, 0x64, 0x77, 0x9D, 0x90, 0x13,
	0x7F, 0x2E, 0x40, 0x80, 0x56, 0xCA, 0x81, 0x43, 0x44, 0x39, 0x70, 0x20,
	0x45, 0x29, 0x78, 0xD7, 0x58, 0x42, 0x25, 0x29, 0x08, 0x5A, 0x53, 0x2A,
	0x81, 0xE1, 0x50, 0xB5, 0x4D, 0xA0, 0x48, 0x89, 0x90, 0x28, 0x5E, 0x27,
	0x4A, 0x04, 0xA9, 0xDA, 0x43, 0x55, 0x05, 0x4E, 0x44, 0x25, 0x28, 0xA9,
	0x54, 0x2E, 0x45, 0x8A, 0x68, 0x1B, 0x7B, 0x6D, 0xC7, 0xA9, 0xA3, 0x54,
	0xCD, 0xEE, 0xDA, 0x8D, 0x9D, 0xA6, 0xB1, 0xB3, 0x33, 0x3F, 0x0E, 0x89,
	0x8B, 0x93, 0xD8, 0x33, 0xB3, 0xB3, 0x6F, 0xC6, 0x36, 0xDA, 0xCF, 0x6D,
	0xE6, 0xBD, 0xF7, 0xFB, 0x7D, 0xDF, 0x6F, 0xDF, 0x7B, 0x33, 0xEF, 0xCF,
	0x2C, 0x34, 0x69, 0xD2, 0xA4, 0x49, 0x93, 0x26, 0x4D, 0x9A, 0xF8, 0x46,
	0x96, 0xCA, 0x71, 0x2E, 0xFF, 0xDE, 0x26, 0x88, 0x6D, 0x07, 0x67, 0xAB,
	0x0A, 0x39, 0x90, 0xCD, 0x40, 0x2B, 0xE8, 0x6A, 0x90, 0x18, 0xA8, 0x0D,
	0x72, 0x19, 0x98, 0x02, 0x1D, 0x55, 0x18, 0x04, 0xEB, 0xED, 0x98, 0xD4,
	0x4E, 0x0F, 0xF6, 0xAF, 0x3B, 0xB7, 0x14, 0x9A, 0xA3, 0x0B, 0x56, 0x9F,
	0x5A, 0xB9, 0xF1, 0xEA, 0x03, 0x88, 0xEE, 0x72, 0x94, 0x1D, 0x02, 0x1B,
	0x83, 0x1B, 0x93, 0x31, 0x45, 0x4F, 0xC4, 0x44, 0x8E, 0x0D, 0x6E, 0x68,
	0x3B, 0x49, 0x9F, 0x38, 0xE6, 0x84, 0xBA, 0x78, 0x0D, 0xDB, 0x41, 0xAE,
	0xE7, 0x62, 0x0A, 0x3B, 0xB1, 0x47, 0xA1, 0x0B, 0x68, 0x0F, 0xC1, 0xC5,
	0xB8, 0x08, 0x03, 0xD6, 0x6C, 0xF2, 0xC0, 0xD9, 0x23, 0xAB, 0xAA, 0x21,
	0xD8, 0xFF, 0x88, 0xD0, 0x82, 0x95, 0xE9, 0x7A, 0x37, 0x6D, 0x59, 0xF1,
	0x27, 0x14, 0x7A, 0x80, 0x7B, 0xC2, 0xF2, 0x33, 0x8F, 0x2B, 0xAA, 0xFC,
	0xD1, 0x8A, 0x5F, 0x7F, 0x7A, 0xE8, 0xF9, 0xF5, 0x95, 0x30, 0x1C, 0x84,
	0x10, 0x2C, 0x95, 0x5C, 0xBE, 0xBA, 0x5B, 0x45, 0x7F, 0x0B, 0xA4, 0xCC,
	0xDB, 0x77, 0x47, 0x60, 0x52, 0xD1, 0x5F, 0x97, 0xA6, 0xD2, 0x07, 0x79,
	0x49, 0x6C, 0xC3, 0xB6, 0xCD, 0xD1, 0x51, 0x98, 0xDC, 0x18, 0xE7, 0xFA,
	0x31, 0x90, 0x2F, 0x99, 0xB4, 0x1B, 0x04, 0x85, 0xD7, 0x9C, 0x58, 0xEC,
	0xA1, 0xD1, 0xE7, 0xD7, 0xBC, 0x63, 0xCA, 0xA6, 0x65, 0xCA, 0x50, 0x26,
	0x5F, 0xFE, 0x66, 0x9C, 0xDA, 0x1B, 0xCB, 0x21, 0x50, 0x00, 0x02, 0x5F,
	0x89, 0xD9, 0xF6, 0x99, 0x4C, 0xF7, 0xA5, 0x6F, 0x9B, 0xB2, 0x69, 0x24,
	0x58, 0xD9, 0xC2, 0xA5, 0x47, 0x45, 0x78, 0x19, 0x58, 0x63, 0xC2, 0x9E,
	0x41, 0x56, 0x8B, 0xCA, 0x8B, 0xD9, 0x42, 0xE5, 0x71, 0x13, 0xC6, 0x1A,
	0xEC, 0x86, 0x2A, 0xD9, 0x42, 0xE5, 0x59, 0xE0, 0x67, 0x26, 0xC4, 0x84,
	0x8A, 0xB0, 0xAF, 0xD4, 0x9F, 0xEA, 0x05, 0xD1, 0xE0, 0x26, 0x1A, 0x20,
	0x5B, 0x28, 0xEF, 0x63, 0x25, 0x04, 0x6A, 0x0E, 0x61, 0x5F, 0xA9, 0x3F,
	0x1D, 0xB8, 0x95, 0x05, 0xEE, 0x86, 0x99, 0x42, 0xA5, 0x97, 0x95, 0x14,
	0x28, 0x00, 0xE5, 0xB1, 0x6C, 0xA1, 0x1C, 0x58, 0x73, 0xA0, 0x96, 0x95,
	0xE9, 0xAE, 0x3C, 0x28, 0xAA, 0x2F, 0x07, 0x2D, 0xBF, 0xC4, 0xA8, 0xC2,
	0x83, 0xC3, 0xC5, 0xF4, 0xF1, 0x7A, 0x0B, 0xD6, 0x5D, 0xD9, 0xCD, 0x5D,
	0xD5, 0x76, 0xCB, 0xB2, 0xFF, 0x0D, 0xD2, 0x56, 0x6F, 0xD9, 0xE5, 0x82,
	0xC0, 0xA4, 0x65, 0xD7, 0x3E, 0x7F, 0xF6, 0xF0, 0xA7, 0xC6, 0xEA, 0x29,
	0x57, 0x5F, 0x37, 0xEC, 0x53, 0x2B, 0x26, 0xFA, 0xC2, 0x4A, 0x0E, 0x14,
	0x80, 0xC2, 0xBD, 0x76, 0x2C, 0x7E, 0x94, 0x3E, 0xAD, 0xAB, 0xFE, 0x75,
	0x65, 0xCE, 0xBE, 0x53, 0xE9, 0x51, 0xD1, 0x2F, 0xD7, 0x27, 0x6D, 0xD9,
	0xB2, 0x3D, 0x37, 0x5E, 0xC9, 0xD7, 0x53, 0xC0, 0x77, 0x37, 0xDC, 0xF2,
	0xF0, 0x54, 0x9B, 0x9D, 0x98, 0x39, 0xCB, 0x12, 0x4C, 0x61, 0x42, 0x64,
	0x42, 0x62, 0xD7, 0x3B, 0xFD, 0xCE, 0x25, 0x7D, 0xB7, 0x2C, 0x27, 0x31,
	0xF3, 0x24, 0x11, 0x04, 0x4A, 0xE0, 0x15, 0x84, 0x5F, 0x0A, 0xBC, 0x12,
	0xB6, 0x2F, 0x60, 0x8D, 0x3A, 0x09, 0xDF, 0xAF, 0x12, 0xBE, 0x5A, 0x56,
	0xAE, 0xE7, 0x62, 0x4A, 0xED, 0xC4, 0x18, 0x61, 0xAF, 0x1E, 0x08, 0xBF,
	0x2F, 0xF5, 0xA7, 0x7F, 0x32, 0x77, 0x99, 0xED, 0x2E, 0x3F, 0x87, 0xF2,
	0xE3, 0x50, 0x7D, 0xC2, 0x74, 0x0B, 0x2D, 0xF7, 0xBD, 0x59, 0x5C, 0x3D,
	0xE1, 0x95, 0xD1, 0x5F, 0xCB, 0xB2, 0x13, 0x7B, 0x88, 0x60, 0x99, 0x45,
	0x54, 0x8E, 0xB9, 0x5D, 0x87, 0x44, 0xEB, 0x8C, 0xCE, 0xEE, 0xF1, 0x93,
	0xD1, 0x3B, 0x58, 0x7D, 0x6A, 0xDD, 0x5C, 0xB8, 0x0B, 0x1D, 0x47, 0x75,
	0x9D, 0xDB, 0x75, 0x68, 0x08, 0x5D, 0x7E, 0x9E, 0x8C, 0x9E, 0x19, 0x72,
	0xE3, 0xD5, 0x07, 0x08, 0x67, 0x85, 0xF3, 0x0E, 0x44, 0x78, 0xEA, 0xC6,
	0xDA, 0xFC, 0x8D, 0x35, 0x7A, 0x11, 0x9E, 0x8A, 0xC4, 0x2F, 0x6C, 0xCC,
	0x5C, 0xA8, 0x7C, 0xCD, 0x2B, 0x9F, 0x77, 0xCB, 0x12, 0xDD, 0x65, 0x44,
	0x91, 0x3F, 0xB6, 0xA9, 0x58, 0x23, 0xD9, 0x42, 0xB9, 0xAC, 0x62, 0x8D,
	0x00, 0xDB, 0xA2, 0x72, 0x2C, 0xCA, 0x77, 0xBC, 0xF2, 0x78, 0x06, 0xCB,
	0x51, 0x76, 0x98, 0x91, 0x53, 0x17, 0x4B, 0xF1, 0x7A, 0xE2, 0x59, 0x4F,
	0xD7, 0x60, 0xE5, 0xF2, 0xEF, 0x6D, 0x6A, 0x6C, 0x17, 0x66, 0x45, 0xB1,
	0x79, 0xCB, 0xF7, 0xDF, 0xBD, 0xCF, 0x2D, 0x83, 0x47, 0xCB, 0x8A, 0x6D,
	0x37, 0x28, 0x66, 0xD9, 0x53, 0x8B, 0x27, 0x5C, 0xEB, 0x1B, 0x77, 0x2F,
	0xEE, 0x6C, 0x8D, 0x70, 0x61, 0xE1, 0x22, 0x70, 0x5C, 0xD0, 0xF7, 0x15,
	0xB2, 0x20, 0xDF, 0x00, 0xEE, 0x8E, 0xCA, 0xF9, 0x4D, 0xB6, 0xBA, 0x25,
	0xBA, 0x06, 0x4B, 0x2D, 0x3A, 0x09, 0xBC, 0xAE, 0x58, 0x17, 0xFB, 0x3F,
	0xDD, 0x9E, 0x7A, 0xE2, 0x64, 0x9F, 0xD4, 0xE6, 0x6E, 0x64, 0xBA, 0xAB,
	0x1B, 0x44, 0x9D, 0xA3, 0xC0, 0xFD, 0x91, 0x28, 0x00, 0x44, 0xE9, 0x74,
	0x4B, 0x77, 0xEF, 0x86, 0x2A, 0x9B, 0x8C, 0xAA, 0x59, 0x10, 0x19, 0x2B,
	0xB5, 0xA7, 0x7A, 0xE7, 0x07, 0x0A, 0x60, 0xB8, 0xBF, 0xED, 0xC2, 0x5D,
	0xB3, 0xF6, 0x4E, 0xE0, 0xCD, 0xF0, 0x35, 0xCC, 0xA1, 0xAE, 0xF5, 0xF5,
	0x7A, 0x1A, 0xAE, 0x32, 0xA8, 0x64, 0x61, 0x54, 0xCB, 0x8B, 0x6D, 0xBF,
	0x9F, 0x39, 0xBA, 0xEE, 0x03, 0xD4, 0xF9, 0x55, 0xE8, 0x1A, 0xFE, 0xC7,
	0x6A, 0xB7, 0x44, 0xAF, 0x60, 0xB5, 0x1A, 0x14, 0xB2, 0x30, 0xC2, 0xE7,
	0x3A, 0xF3, 0xE5, 0x2F, 0x2E, 0x96, 0x3C, 0x93, 0x4C, 0xBE, 0x0A, 0x11,
	0x0D, 0x06, 0x1E, 0xF5, 0xF5, 0x18, 0xE0, 0xB9, 0xCB, 0xA0, 0x90, 0x45,
	0x35, 0x38, 0xC2, 0xAB, 0x99, 0xFC, 0xA5, 0x23, 0x22, 0xF2, 0xA1, 0x5A,
	0xD6, 0xE1, 0xE1, 0x3F, 0xB5, 0x7D, 0xD4, 0xF5, 0x6A, 0xF6, 0xD5, 0x99,
	0x24, 0x89, 0xA8, 0x9E, 0x32, 0xAE, 0x0F, 0x14, 0xAF, 0x96, 0x75, 0xD5,
	0xA0, 0x10, 0x37, 0x5A, 0x45, 0xE4, 0x11, 0x60, 0xCF, 0xB5, 0xF7, 0xAF,
	0x8E, 0xCE, 0x4F, 0xB8, 0x5B, 0x93, 0x1D, 0x11, 0x69, 0x00, 0x8F, 0xFA,
	0x7A, 0x04, 0x4B, 0x3E, 0x30, 0xA9, 0xC4, 0x13, 0xE1, 0xF5, 0x0B, 0x2F,
	0xB5, 0x7F, 0x78, 0xCB, 0x3D, 0x47, 0xBF, 0x1E, 0xA1, 0x82, 0x69, 0xB7,
	0x44, 0x8F, 0x60, 0xE9, 0x65, 0x93, 0x4A, 0xBC, 0x10, 0xE5, 0x9F, 0xB7,
	0xDF, 0xB3, 0x24, 0xD2, 0x60, 0xB9, 0xD6, 0xD7, 0x3D, 0x58, 0xA2, 0xA3,
	0xAE, 0xE9, 0x86, 0xB1, 0x95, 0x93, 0xB7, 0xDE, 0x51, 0x51, 0x25, 0xBA,
	0x60, 0xA9, 0x8C, 0xB8, 0x25, 0xBB, 0x06, 0x4B, 0x1C, 0x06, 0xCD, 0xAA,
	0x71, 0x65, 0x66, 0x66, 0xFA, 0xDA, 0xEB, 0xF3, 0x6F, 0x64, 0xBA, 0x26,
	0xB6, 0x02, 0x6B, 0x23, 0x53, 0x20, 0xEE, 0xF5, 0x75, 0x0D, 0x96, 0x23,
	0xD6, 0x5B, 0x66, 0xD5, 0xB8, 0xF2, 0xAF, 0x3B, 0xC6, 0xAB, 0x58, 0xA4,
	0x5D, 0x10, 0x55, 0x7D, 0xDB, 0x2D, 0xDD, 0x35, 0x58, 0x96, 0xDA, 0xAF,
	0x99, 0x95, 0xB3, 0x38, 0x22, 0xFC, 0xE3, 0x0E, 0xFF, 0xE8, 0xFD, 0x51,
	0xF9, 0x07, 0xB0, 0x25, 0x7E, 0xDA, 0x2D, 0xDD, 0x35, 0x58, 0x43, 0x03,
	0x6B, 0x47, 0x41, 0xC6, 0x8C, 0x2A, 0x5A, 0x04, 0xDB, 0xB9, 0x7D, 0x70,
	0x8F, 0x78, 0xBC, 0x82, 0xE1, 0x73, 0xC5, 0x7B, 0xCF, 0xBB, 0x65, 0xF0,
	0x5C, 0xFC, 0x53, 0xF4, 0x84, 0x39, 0x3D, 0x8B, 0x93, 0xA8, 0x25, 0xCF,
	0xCC, 0xBF, 0xBE, 0x39, 0x5E, 0xB5, 0x02, 0x91, 0x9C, 0x44, 0x46, 0xF0,
	0xAC, 0xA7, 0x67, 0xB0, 0x62, 0x12, 0xC9, 0x0E, 0x8B, 0xDE, 0xF3, 0xB1,
	0xD6, 0xA9, 0xF9, 0x37, 0xAE, 0x5D, 0xB9, 0x7A, 0xAE, 0x34, 0x95, 0x5A,
	0x25, 0xB1, 0xEB, 0x6B, 0x41, 0xFF, 0x1E, 0xBA, 0x00, 0x9B, 0x17, 0xBC,
	0xF2, 0x78, 0x4F, 0x23, 0xFA, 0xD4, 0xCA, 0x8E, 0x57, 0xC6, 0x08, 0x79,
	0xD3, 0x42, 0x1D, 0xBE, 0x30, 0x7C, 0x28, 0xFD, 0xC6, 0x42, 0x69, 0x9D,
	0xF9, 0xF2, 0x7A, 0x47, 0x18, 0xC7, 0xE0, 0xB1, 0xCE, 0x5B, 0x7C, 0xC3,
	0xF9, 0xE1, 0x62, 0xAA, 0xC3, 0xEB, 0xA0, 0x9B, 0x8F, 0xAD, 0x30, 0x71,
	0x44, 0x18, 0x30, 0xA6, 0x6C, 0x31, 0x62, 0xFA, 0xE8, 0x62, 0x49, 0x83,
	0x03, 0xE9, 0x8B, 0x40, 0x98, 0xB3, 0x89, 0x01, 0x3F, 0x27, 0x02, 0x7D,
	0xFD, 0x52, 0xD6, 0x6C, 0xF2, 0x00, 0x70, 0xA5, 0x61, 0x49, 0x2E, 0x88,
	0xCA, 0xEE, 0x5C, 0x77, 0xE5, 0x91, 0x85, 0xD2, 0x72, 0x85, 0xCA, 0xB7,
	0x08, 0x6F, 0x05, 0x64, 0x3A, 0x49, 0xCB, 0x41, 0x3F, 0x19, 0x7D, 0xCF,
	0xE6, 0x33, 0xF9, 0xF2, 0x7E, 0x11, 0x7E, 0x1A, 0x5C, 0x93, 0x4F, 0x94,
	0xBF, 0x8A, 0x48, 0xD1, 0xC6, 0xF9, 0x8F, 0x05, 0xEB, 0x11, 0xD9, 0x89,
	0xB2, 0x17, 0x48, 0x86, 0xE1, 0x4E, 0xD0, 0x67, 0x86, 0x8A, 0x9F, 0xFC,
	0xB9, 0xBF, 0xBC, 0x3E, 0xF9, 0xFF, 0x3C, 0x45, 0xA3, 0x55, 0x89, 0xD5,
	0xB6, 0x18, 0x3F, 0x45, 0x73, 0xF6, 0xC8, 0xAA, 0xAA, 0x08, 0x4F, 0x06,
	0x17, 0xB6, 0xFC, 0x50, 0x78, 0xBC, 0x9E, 0x4F, 0x57, 0xEA, 0x7A, 0xBA,
	0x0C, 0x6D, 0x48, 0xF5, 0x2B, 0x44, 0xF6, 0x56, 0x1F, 0x32, 0xA7, 0x86,
	0x8B, 0xE9, 0x43, 0xF5, 0x14, 0xA8, 0xF3, 0x98, 0xA4, 0x38, 0xEA, 0x58,
	0xBB, 0x40, 0x43, 0xFD, 0xFA, 0x2A, 0x6C, 0x04, 0x26, 0x6B, 0xC4, 0x77,
	0xD7, 0x7B, 0x26, 0xBE, 0xEE, 0xF7, 0x96, 0x91, 0x43, 0x6D, 0xE3, 0x8A,
	0x74, 0x11, 0xDD, 0xBA, 0xB8, 0x69, 0x1C, 0x85, 0xDD, 0x5E, 0x53, 0x9B,
	0x85, 0x08, 0xF4, 0x92, 0x37, 0x5C, 0x4C, 0x1F, 0x17, 0x64, 0x6F, 0x90,
	0xB2, 0x4B, 0x8D, 0x28, 0x8F, 0x95, 0x8A, 0xE9, 0xBF, 0x05, 0x29, 0x1B,
	0xF8, 0x8D, 0x78, 0xA8, 0x98, 0x3A, 0x88, 0xB0, 0x2F, 0x68, 0xF9, 0xA5,
	0x40, 0x54, 0x9F, 0x1E, 0x1A, 0x48, 0xFF, 0x2E, 0x68, 0xF9, 0x86, 0xA6,
	0x0F, 0xA5, 0xFE, 0x54, 0xEF, 0x4A, 0x09, 0x98, 0xA0, 0xCF, 0x0C, 0x0D,
	0xA4, 0x7F, 0xD1, 0x98, 0x0D, 0x03, 0x64, 0xBB, 0x2F, 0xED, 0x45, 0xE5,
	0x39, 0x42, 0x9A, 0xBB, 0x35, 0x88, 0xA2, 0xF4, 0x96, 0x06, 0xD2, 0x0D,
	0xFF, 0xA8, 0xC6, 0xF6, 0xE3, 0xB2, 0x3F, 0x28, 0xEF, 0xC4, 0xD1, 0xC3,
	0xCB, 0xEB, 0x83, 0x02, 0xAD, 0x82, 0x7C, 0x2F, 0xE8, 0x18, 0x75, 0x3B,
	0x46, 0x37, 0x2F, 0x37, 0xF5, 0x4C, 0x7C, 0x26, 0x66, 0xDB, 0x7F, 0x06,
	0x96, 0xC3, 0x51, 0xA5, 0x53, 0x8E, 0x63, 0x7D, 0x77, 0xE4, 0x50, 0xDB,
	0xB8, 0x29, 0x83, 0x61, 0x7E, 0x23, 0xBD, 0x1F, 0x48, 0x9B, 0xB7, 0xEF,
	0xC9, 0x04, 0xE8, 0x6F, 0x4A, 0xED, 0xE9, 0x03, 0xA6, 0xFF, 0xC2, 0x20,
	0xB4, 0x6D, 0xF1, 0x2D, 0x0F, 0x4F, 0xB5, 0xD9, 0x2D, 0x33, 0xBD, 0x28,
	0x3F, 0x04, 0x3E, 0x1E, 0x96, 0x9F, 0x79, 0x4C, 0x0B, 0xFA, 0x87, 0x04,
	0xC9, 0x67, 0xFD, 0x9C, 0x69, 0x0F, 0x42, 0xE8, 0x67, 0x08, 0xB6, 0x15,
	0x2E, 0xAF, 0x99, 0x61, 0xF6, 0x47, 0x40, 0x3E, 0x8C, 0x23, 0x97, 0x0A,
	0xE7, 0x81, 0x81, 0x24, 0x2D, 0x07, 0xC3, 0x0A, 0xD2, 0x1C, 0x91, 0xFE,
	0x63, 0x48, 0xE6, 0x7C, 0xE5, 0xAB, 0x22, 0x3C, 0x84, 0xB0, 0x03, 0x68,
	0xE4, 0xEC, 0xD7, 0x08, 0x70, 0x02, 0xF4, 0x2F, 0xA5, 0x62, 0xFA, 0x54,
	0x23, 0x9F, 0xF2, 0xD6, 0xC3, 0x92, 0x7D, 0x5C, 0xD9, 0x51, 0x98, 0xDC,
	0x18, 0xD3, 0xDA, 0x76, 0x2C, 0xF9, 0xAC, 0x28, 0x9D, 0x88, 0x76, 0xA0,
	0x7C, 0x82, 0x1B, 0x67, 0xC2, 0xE2, 0x40, 0x0D, 0x98, 0x12, 0x98, 0x54,
	0x95, 0x73, 0x6A, 0x31, 0x84, 0xA3, 0x6F, 0xD9, 0x12, 0x3F, 0x1D, 0x64,
	0xAA, 0xD2, 0xA4, 0x49, 0x93, 0x26, 0x4D, 0x9A, 0x34, 0x69, 0x12, 0x31,
	0xFF, 0x05, 0x5C, 0xBF, 0x36, 0x36, 0xF4, 0x4D, 0x49, 0xA5, 0x00, 0x00,
	0x00, 0x00, 0x49, 0x45, 0x4E, 0x44, 0xAE, 0x42, 0x60, 0x82
};

static unsigned char error_[1616] = {
	0x89, 0x50, 0x4E, 0x47, 0x0D, 0x0A, 0x1A, 0x0A, 0x00, 0x00, 0x00, 0x0D,
	0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x4B, 0x00, 0x00, 0x00, 0x4B,
	0x08, 0x06, 0x00, 0x00, 0x00, 0x38, 0x4E, 0x7A, 0xEA, 0x00, 0x00, 0x00,
	0x04, 0x73, 0x42, 0x49, 0x54, 0x08, 0x08, 0x08, 0x08, 0x7C, 0x08, 0x64,
	0x88, 0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x2E,
	0x23, 0x00, 0x00, 0x2E, 0x23, 0x01, 0x78, 0xA5, 0x3F, 0x76, 0x00, 0x00,
	0x00, 0x19, 0x74, 0x45, 0x58, 0x74, 0x53, 0x6F, 0x66, 0x74, 0x77, 0x61,
	0x72, 0x65, 0x00, 0x77, 0x77, 0x77, 0x2E, 0x69, 0x6E, 0x6B, 0x73, 0x63,
	0x61, 0x70, 0x65, 0x2E, 0x6F, 0x72, 0x67, 0x9B, 0xEE, 0x3C, 0x1A, 0x00,
	0x00, 0x05, 0xCD, 0x49, 0x44, 0x41, 0x54, 0x78, 0x9C, 0xED, 0x9C, 0x4D,
	0x6C, 0x54, 0x55, 0x14, 0xC7, 0xFF, 0xE7, 0xBE, 0x41, 0xA3, 0x48, 0xE7,
	0xA3, 0xD4, 0xB0, 0xC1, 0x62, 0x9B, 0xF0, 0x61, 0x5C, 0x19, 0x36, 0x96,
	0x6A, 0x4A, 0xDF, 0x2B, 0x1F, 0x89, 0x81, 0x8D, 0x0B, 0xC1, 0x10, 0x03,
	0x0B, 0x12, 0xA3, 0xE2, 0x42, 0xA1, 0x8A, 0x2E, 0x06, 0x56, 0x40, 0x51,
	0x17, 0x34, 0x26, 0x26, 0xA6, 0x9A, 0x18, 0x85, 0xC8, 0x82, 0x44, 0x62,
	0x42, 0xA7, 0xB7, 0xA4, 0x91, 0xCE, 0xCE, 0x48, 0x30, 0x28, 0x62, 0xA4,
	0xB4, 0xB2, 0x10, 0x89, 0xED, 0x9B, 0xB6, 0x28, 0x8B, 0xBE, 0x79, 0xC7,
	0xC5, 0xB4, 0x06, 0xD4, 0xDC, 0xF7, 0x31, 0xF7, 0xBD, 0x99, 0x29, 0xEF,
	0xB7, 0x9B, 0xDC, 0x7B, 0xCF, 0x39, 0xF7, 0xDF, 0xFB, 0x6E, 0xE7, 0xBE,
	0x7B, 0xCE, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0xBE,
	0xA1, 0x5A, 0x39, 0x2E, 0x59, 0x56, 0x1B, 0x03, 0x9D, 0x04, 0xAC, 0x63,
	0x60, 0x35, 0x98, 0xDB, 0x41, 0xB4, 0x0C, 0x40, 0x1A, 0x80, 0x01, 0xA0,
	0x0C, 0x60, 0x1A, 0xC0, 0x0C, 0x80, 0x31, 0x22, 0xBA, 0xCA, 0xC0, 0x15,
	0x22, 0x1A, 0xCD, 0x14, 0x0A, 0xD7, 0x6B, 0x11, 0x73, 0x6C, 0x62, 0x71,
	0x3E, 0x2F, 0xA6, 0x8B, 0xC5, 0x8D, 0x0C, 0xEC, 0x00, 0xB3, 0x05, 0xA0,
	0x35, 0xBC, 0x31, 0x1E, 0x07, 0x20, 0xC9, 0x30, 0x4E, 0xA6, 0x3B, 0x3A,
	0x46, 0x28, 0x9F, 0x77, 0x75, 0xC5, 0xA9, 0x22, 0x72, 0xB1, 0x66, 0xBA,
	0xBA, 0x96, 0x97, 0x0D, 0xE3, 0x35, 0x10, 0xED, 0x06, 0xB0, 0x32, 0x02,
	0x17, 0x37, 0x00, 0x0C, 0x18, 0xCC, 0x27, 0x9A, 0x86, 0x87, 0x27, 0x23,
	0xB0, 0xFF, 0x0F, 0x91, 0x89, 0x35, 0xBB, 0x75, 0x6B, 0x8B, 0x33, 0x37,
	0xF7, 0x36, 0x80, 0xBD, 0x00, 0x96, 0x46, 0xE5, 0xE7, 0x2E, 0x6E, 0x83,
	0xF9, 0x23, 0xA3, 0x5C, 0x3E, 0xD2, 0x34, 0x32, 0xF2, 0x47, 0x14, 0x0E,
	0xB4, 0x8B, 0xC5, 0x00, 0x95, 0x2C, 0x6B, 0x17, 0x80, 0xF7, 0x00, 0x2C,
	0xD7, 0x6D, 0xDF, 0x07, 0x36, 0x88, 0x0E, 0x65, 0x32, 0x99, 0x7E, 0x3A,
	0x7D, 0xBA, 0xAC, 0xD3, 0xB0, 0x56, 0xB1, 0xEC, 0xEE, 0xEE, 0x56, 0x12,
	0xE2, 0x24, 0x03, 0x4F, 0xEB, 0xB4, 0x1B, 0x92, 0x22, 0x03, 0x3B, 0x73,
	0x52, 0xFE, 0xAA, 0xCB, 0xA0, 0xD0, 0x65, 0xC8, 0xEE, 0xE9, 0xD9, 0x06,
	0x21, 0xBE, 0xAB, 0x13, 0xA1, 0x00, 0x60, 0x03, 0x01, 0xDF, 0x97, 0x4C,
	0xF3, 0x79, 0x5D, 0x06, 0xB5, 0x88, 0x65, 0xF7, 0xF4, 0xBC, 0x0E, 0xE6,
	0x33, 0x00, 0x72, 0x3A, 0xEC, 0x69, 0x24, 0xCD, 0x44, 0x5F, 0x96, 0x4C,
	0xF3, 0x80, 0x0E, 0x63, 0x55, 0x3D, 0x86, 0x0C, 0x90, 0x6D, 0x59, 0xC7,
	0x08, 0x78, 0x53, 0x47, 0x30, 0x51, 0xC2, 0x40, 0x5F, 0x56, 0xCA, 0x5E,
	0x02, 0x38, 0xAC, 0x8D, 0xAA, 0xC4, 0x9A, 0xB2, 0xAC, 0xBE, 0x46, 0x10,
	0x6A, 0x01, 0x06, 0xFA, 0x72, 0x52, 0x86, 0x5E, 0x65, 0xA1, 0x1F, 0xC3,
	0x52, 0x4F, 0x4F, 0x6F, 0x23, 0x09, 0x05, 0x00, 0x04, 0xEC, 0x2F, 0x59,
	0x56, 0xE8, 0x98, 0x43, 0xAD, 0x2C, 0xDB, 0xB2, 0xB6, 0x03, 0x38, 0x13,
	0x76, 0x7C, 0x8D, 0x61, 0x16, 0x62, 0x7B, 0xAE, 0x50, 0x38, 0x1B, 0x74,
	0x60, 0xE0, 0xC9, 0x4E, 0x6E, 0xDA, 0xB4, 0x52, 0xB8, 0xEE, 0x45, 0x00,
	0xCD, 0x41, 0xC7, 0xD6, 0x11, 0x36, 0x52, 0xA9, 0xA7, 0xB2, 0xE7, 0xCE,
	0x8D, 0x07, 0x19, 0x14, 0xE8, 0x31, 0xE4, 0x7C, 0x5E, 0x08, 0xE6, 0x53,
	0x68, 0x6C, 0xA1, 0x00, 0x20, 0x0B, 0xC7, 0xF9, 0x8C, 0xF3, 0xF9, 0x40,
	0xF3, 0x0F, 0xD4, 0xD9, 0x2E, 0x16, 0xF7, 0x82, 0xB9, 0x23, 0x58, 0x5C,
	0x75, 0x4B, 0xE7, 0x74, 0xB1, 0xB8, 0x27, 0xC8, 0x00, 0xDF, 0x8F, 0xE1,
	0x8C, 0x69, 0x36, 0x97, 0x89, 0x7E, 0x42, 0x6D, 0x8E, 0x30, 0x51, 0x31,
	0x65, 0x38, 0xCE, 0x1A, 0xBF, 0x67, 0x49, 0xDF, 0x2B, 0xAB, 0x4C, 0xF4,
	0x0E, 0x16, 0x97, 0x50, 0x00, 0x90, 0x73, 0x96, 0x2C, 0xD9, 0xEF, 0xB7,
	0xB3, 0xAF, 0x95, 0x35, 0xBF, 0xAA, 0xC6, 0x01, 0x3C, 0x12, 0x36, 0xAA,
	0x3A, 0xE6, 0x4F, 0xC3, 0x71, 0x56, 0xF9, 0x59, 0x5D, 0xBE, 0x56, 0x56,
	0x19, 0xD8, 0x87, 0xF8, 0x84, 0xBA, 0x03, 0xA2, 0x1F, 0x00, 0xDC, 0x89,
	0xC9, 0xDF, 0xD2, 0x72, 0x2A, 0xF5, 0x8A, 0x9F, 0x8E, 0x9E, 0x62, 0x71,
	0x3E, 0x2F, 0xE6, 0x5F, 0xDC, 0xC5, 0xC1, 0xC5, 0x39, 0xC7, 0x69, 0xCB,
	0x0E, 0x0D, 0x3D, 0xE9, 0x18, 0x46, 0x3B, 0x98, 0x2F, 0xC5, 0xE4, 0x77,
	0xB7, 0x9F, 0xFF, 0x8C, 0x9E, 0x1D, 0xA6, 0x8B, 0xC5, 0x8D, 0x88, 0xE6,
	0x0D, 0xE7, 0x7F, 0x60, 0xA2, 0x77, 0x1F, 0x1D, 0x19, 0xB9, 0x09, 0x00,
	0x2D, 0x83, 0x83, 0xBF, 0x71, 0x65, 0x9F, 0x8C, 0x83, 0xD6, 0xD2, 0xE8,
	0xE8, 0xB3, 0x5E, 0x9D, 0xBC, 0x57, 0x16, 0xB0, 0x43, 0x4F, 0x3C, 0xDE,
	0xB8, 0x44, 0x63, 0xF7, 0x7C, 0x16, 0xE2, 0x5A, 0x5C, 0xBE, 0x99, 0xE8,
	0x05, 0xAF, 0x3E, 0xDE, 0x7B, 0x56, 0xE5, 0x72, 0x61, 0xD1, 0x43, 0x3E,
	0xE6, 0xA9, 0x14, 0xAB, 0x64, 0x59, 0x6D, 0xA8, 0xE6, 0x16, 0xA6, 0xB1,
	0x68, 0xB7, 0xB7, 0x6C, 0x59, 0xA5, 0xEA, 0xA0, 0x14, 0x8B, 0x81, 0x4E,
	0xAD, 0xE1, 0xD4, 0x39, 0x5C, 0x2E, 0x2B, 0xE7, 0xAB, 0x14, 0x8B, 0x80,
	0x75, 0x7A, 0xC3, 0xA9, 0x6F, 0x04, 0xF3, 0x5A, 0x65, 0xBB, 0xAA, 0x91,
	0x99, 0xD7, 0xE8, 0x0D, 0xA7, 0xBE, 0x61, 0x22, 0xE5, 0x7C, 0xBD, 0x36,
	0xF8, 0x36, 0x8D, 0xB1, 0xD4, 0x3F, 0xCC, 0xED, 0xAA, 0x66, 0x2F, 0xB1,
	0xD2, 0x1A, 0x43, 0x69, 0x04, 0x94, 0xF3, 0x55, 0x8B, 0x45, 0xB4, 0x18,
	0xCF, 0x82, 0x2A, 0x96, 0xA9, 0x1A, 0xBD, 0x56, 0xD6, 0x43, 0x1A, 0x03,
	0x69, 0x04, 0x1E, 0x56, 0x35, 0x7A, 0x89, 0x15, 0xD7, 0x61, 0xB6, 0x5E,
	0xF8, 0x4B, 0xD5, 0xE8, 0x25, 0xD6, 0xAC, 0xC6, 0x40, 0x1A, 0x01, 0xE5,
	0x7C, 0xD5, 0x62, 0x31, 0xCF, 0x68, 0x0D, 0xC5, 0x03, 0xC1, 0xDC, 0xA6,
	0xFA, 0x1C, 0x03, 0xD3, 0xAA, 0x46, 0xAF, 0x0D, 0x3E, 0xB6, 0x83, 0x2C,
	0x00, 0x10, 0xF3, 0xA1, 0x5B, 0x5D, 0x5D, 0x2B, 0x00, 0xE0, 0x56, 0x57,
	0xD7, 0x0A, 0x62, 0x3E, 0x1C, 0xA7, 0x7F, 0xAF, 0xF9, 0xA6, 0xD4, 0x63,
	0xE9, 0x2A, 0x73, 0xE8, 0xDB, 0xEE, 0x30, 0xAC, 0x5F, 0x92, 0x4A, 0x8D,
	0xDB, 0x96, 0x35, 0x01, 0x60, 0x15, 0x80, 0x07, 0xE2, 0x74, 0x4E, 0xCC,
	0x57, 0x55, 0xED, 0x4A, 0xB1, 0x18, 0xB8, 0xA2, 0x37, 0x1C, 0x5F, 0x3C,
	0x08, 0x60, 0x75, 0x0D, 0xFC, 0xC2, 0x65, 0x56, 0xCE, 0x57, 0x7D, 0x36,
	0x64, 0x2E, 0xEA, 0x0D, 0xA7, 0xBE, 0x21, 0xE6, 0x51, 0x55, 0xBB, 0x52,
	0xAC, 0x8C, 0x94, 0x63, 0xF3, 0xC9, 0xAE, 0xF7, 0x03, 0xBF, 0x64, 0xCF,
	0x9F, 0x9F, 0x50, 0x75, 0xF0, 0x73, 0x61, 0x21, 0x35, 0x05, 0x13, 0x84,
	0x48, 0x72, 0x42, 0x55, 0x90, 0x8F, 0x79, 0x7A, 0x8A, 0x45, 0x86, 0x71,
	0x52, 0x4F, 0x38, 0xBE, 0xB8, 0x4C, 0x42, 0xB4, 0x65, 0xA5, 0x6C, 0x21,
	0xA0, 0x1D, 0xC0, 0xE5, 0xB8, 0x1C, 0x33, 0xD1, 0x29, 0xAF, 0x3E, 0x9E,
	0x62, 0xA5, 0x3B, 0x3A, 0x46, 0x50, 0x49, 0x9F, 0x8E, 0x1C, 0x16, 0xE2,
	0xE0, 0x42, 0x41, 0x40, 0x46, 0xCA, 0x31, 0xC4, 0x77, 0x61, 0x31, 0x91,
	0x19, 0x1A, 0xFA, 0xC6, 0xAB, 0x93, 0xF7, 0xCA, 0xAA, 0x24, 0xE4, 0x0F,
	0x68, 0x09, 0xC9, 0x03, 0xC3, 0x71, 0x7E, 0xBF, 0xFB, 0xB3, 0x28, 0x97,
	0x6F, 0xC6, 0xE1, 0x17, 0x44, 0x03, 0x7E, 0x32, 0x02, 0x7D, 0x5D, 0xB2,
	0x1A, 0xCC, 0x27, 0x00, 0xDC, 0xAE, 0x3A, 0x28, 0x0F, 0x5C, 0x21, 0xEE,
	0xB9, 0x49, 0x72, 0x89, 0x76, 0x46, 0xED, 0x13, 0xC0, 0xAC, 0x10, 0xA2,
	0xDF, 0x4F, 0x47, 0xDF, 0x89, 0x21, 0xB6, 0x69, 0x1E, 0x07, 0xD1, 0x1B,
	0xE1, 0x63, 0xF2, 0xCD, 0x57, 0x60, 0xFE, 0x16, 0x44, 0xEB, 0x01, 0x6C,
	0x8B, 0xDA, 0x19, 0x03, 0x47, 0x73, 0x52, 0xBE, 0xE5, 0xA7, 0xEF, 0xFD,
	0x9E, 0x45, 0x33, 0x69, 0x38, 0xCE, 0x5A, 0xED, 0x59, 0x34, 0x4D, 0xC3,
	0xC3, 0x93, 0x04, 0xC4, 0xB5, 0xE1, 0xC6, 0xC5, 0x81, 0x20, 0xA5, 0x2B,
	0x81, 0x92, 0xD9, 0xD2, 0x9D, 0x9D, 0x1F, 0x03, 0x58, 0x2C, 0xDF, 0xEA,
	0x2F, 0x64, 0xA4, 0xFC, 0x24, 0xC8, 0x80, 0x40, 0x62, 0x51, 0x3E, 0xEF,
	0xCE, 0x6F, 0xC2, 0x91, 0x56, 0x5F, 0xC5, 0x80, 0x0D, 0xD7, 0xDD, 0x15,
	0x34, 0x27, 0x3E, 0x70, 0x6A, 0x77, 0x73, 0xA1, 0x70, 0x83, 0x85, 0xD8,
	0x8D, 0x2A, 0x92, 0xEF, 0x6B, 0x8C, 0xCB, 0xC0, 0x2E, 0xAF, 0xA3, 0xCD,
	0xFF, 0x11, 0x2A, 0x0F, 0x3E, 0x57, 0x28, 0x9C, 0xE5, 0x4A, 0xCE, 0x56,
	0xE3, 0xC1, 0xBC, 0x3F, 0x27, 0xE5, 0xD7, 0x61, 0x86, 0x86, 0x2E, 0x1A,
	0xC8, 0x49, 0xD9, 0xCF, 0x40, 0x5F, 0xD8, 0xF1, 0xB5, 0x80, 0x80, 0x23,
	0xD9, 0xE1, 0xE1, 0xF7, 0xC3, 0x8E, 0xAF, 0xAA, 0xD0, 0x29, 0x2B, 0x65,
	0x6F, 0xA3, 0x08, 0xC6, 0xC0, 0xD1, 0xB4, 0x94, 0x07, 0xAB, 0xB1, 0xA1,
	0xA5, 0x42, 0xC2, 0x36, 0xCD, 0x7D, 0x20, 0xFA, 0x00, 0x1A, 0x4B, 0xF2,
	0x34, 0xC2, 0x04, 0xF4, 0x66, 0xA4, 0xAC, 0xFA, 0x8F, 0xAA, 0xAD, 0x9C,
	0x64, 0xCA, 0x34, 0x9F, 0x23, 0xA2, 0x4F, 0x51, 0x5F, 0x05, 0x05, 0x93,
	0x0C, 0xBC, 0x14, 0x76, 0x8F, 0xFA, 0x37, 0x5A, 0x6B, 0x6F, 0xA6, 0x2C,
	0xEB, 0x31, 0x02, 0x3E, 0x47, 0x7D, 0xA4, 0x2A, 0x5D, 0x70, 0x85, 0x78,
	0xB1, 0xB9, 0x50, 0xD0, 0xF6, 0xC6, 0x24, 0xCA, 0x1A, 0xE9, 0xE3, 0x00,
	0x5A, 0x74, 0xDB, 0xF7, 0xC1, 0x14, 0x88, 0x0E, 0x67, 0x36, 0x6C, 0x38,
	0xA1, 0xFB, 0x27, 0x0C, 0x22, 0xAB, 0xEA, 0x9A, 0x31, 0xCD, 0x66, 0x87,
	0xA8, 0x97, 0x80, 0x97, 0x11, 0x4F, 0x5A, 0xF8, 0x2C, 0x03, 0x1F, 0x1A,
	0x86, 0x71, 0x2C, 0x3D, 0x38, 0x38, 0x15, 0x85, 0x83, 0xC8, 0x4B, 0xE0,
	0xA6, 0x37, 0x6F, 0xCE, 0xB9, 0xAE, 0xFB, 0x2A, 0x98, 0xF7, 0x20, 0x9A,
	0x94, 0xCB, 0x09, 0x10, 0x0D, 0x08, 0x21, 0xFA, 0xA3, 0x12, 0x69, 0x81,
	0x58, 0x7F, 0x31, 0xA4, 0x54, 0x2C, 0x3E, 0xC3, 0xCC, 0x3B, 0x09, 0xB0,
	0x50, 0x5D, 0xEE, 0xD7, 0x35, 0x26, 0x92, 0x44, 0xF4, 0x45, 0xA6, 0x50,
	0xB8, 0x50, 0x4D, 0x29, 0x6F, 0x10, 0x6A, 0x56, 0x5C, 0x69, 0x77, 0x77,
	0xB7, 0x32, 0x51, 0xA7, 0x00, 0x9E, 0x98, 0xCF, 0xB8, 0x7B, 0x1C, 0x40,
	0x06, 0x40, 0x13, 0x2A, 0xF7, 0x99, 0x0E, 0x2A, 0xBF, 0x43, 0x63, 0x83,
	0xE8, 0x3A, 0xB9, 0xEE, 0xCF, 0x2E, 0xF0, 0x23, 0x31, 0x8F, 0x86, 0x39,
	0xAA, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0xC4, 0xCC, 0xDF,
	0x99, 0xE6, 0xCF, 0x40, 0x03, 0x6C, 0xA6, 0x08, 0x00, 0x00, 0x00, 0x00,
	0x49, 0x45, 0x4E, 0x44, 0xAE, 0x42, 0x60, 0x82
};

static unsigned char success_[1933] = {
	0x89, 0x50, 0x4E, 0x47, 0x0D, 0x0A, 0x1A, 0x0A, 0x00, 0x00, 0x00, 0x0D,
	0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x4B, 0x00, 0x00, 0x00, 0x4B,
	0x08, 0x06, 0x00, 0x00, 0x00, 0x38, 0x4E, 0x7A, 0xEA, 0x00, 0x00, 0x00,
	0x04, 0x73, 0x42, 0x49, 0x54, 0x08, 0x08, 0x08, 0x08, 0x7C, 0x08, 0x64,
	0x88, 0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x2E,
	0x23, 0x00, 0x00, 0x2E, 0x23, 0x01, 0x78, 0xA5, 0x3F, 0x76, 0x00, 0x00,
	0x00, 0x19, 0x74, 0x45, 0x58, 0x74, 0x53, 0x6F, 0x66, 0x74, 0x77, 0x61,
	0x72, 0x65, 0x00, 0x77, 0x77, 0x77, 0x2E, 0x69, 0x6E, 0x6B, 0x73, 0x63,
	0x61, 0x70, 0x65, 0x2E, 0x6F, 0x72, 0x67, 0x9B, 0xEE, 0x3C, 0x1A, 0x00,
	0x00, 0x07, 0x0A, 0x49, 0x44, 0x41, 0x54, 0x78, 0x9C, 0xED, 0x9C, 0x59,
	0x6C, 0x54, 0x55, 0x18, 0xC7, 0xFF, 0xDF, 0x9D, 0x85, 0xC5, 0xA5, 0x0A,
	0xF5, 0x81, 0x44, 0x2D, 0x73, 0x07, 0x0B, 0x48, 0x8C, 0x46, 0x89, 0xA4,
	0x33, 0x53, 0x09, 0x0A, 0x3E, 0x18, 0x23, 0x2F, 0xC6, 0x58, 0xA0, 0xD3,
	0xB4, 0x12, 0x12, 0x5C, 0x50, 0x12, 0xA1, 0x2A, 0x68, 0xEA, 0x12, 0x41,
	0x01, 0x7D, 0x80, 0x48, 0x44, 0xE9, 0x32, 0x8D, 0x40, 0x44, 0xA3, 0x89,
	0xD1, 0x17, 0x35, 0xA8, 0xED, 0x6D, 0x35, 0x51, 0x8C, 0x18, 0xB1, 0x10,
	0xE6, 0x4E, 0x6B, 0x91, 0x98, 0x68, 0xA9, 0xEC, 0x32, 0xCB, 0xFD, 0x7C,
	0x28, 0x85, 0x66, 0x3A, 0x73, 0xD7, 0x73, 0x67, 0x06, 0x32, 0xBF, 0xB7,
	0x9E, 0xE5, 0x3B, 0xFF, 0xFB, 0xF5, 0x9C, 0x7B, 0xCF, 0xF2, 0x9D, 0x01,
	0xCA, 0x94, 0x29, 0x53, 0xA6, 0x4C, 0x99, 0x32, 0x65, 0x4C, 0x43, 0xC5,
	0x6A, 0xB8, 0xBA, 0xAB, 0x49, 0xCE, 0x78, 0x32, 0x11, 0xD6, 0x78, 0x36,
	0x24, 0x54, 0x83, 0x11, 0x64, 0xF0, 0x35, 0x00, 0x55, 0x10, 0xE0, 0x61,
	0x20, 0x03, 0xF0, 0x09, 0x02, 0x9D, 0x04, 0x41, 0x05, 0xF8, 0x10, 0x33,
	0x7E, 0xD7, 0xD2, 0xE8, 0x1E, 0x98, 0xDF, 0x99, 0x28, 0x86, 0xE6, 0xC2,
	0x39, 0x8B, 0x5B, 0x24, 0xB9, 0x37, 0xB1, 0x00, 0x40, 0x1D, 0x98, 0x17,
	0x02, 0xA8, 0x72, 0x60, 0xAD, 0x9F, 0x80, 0xAF, 0x00, 0xDE, 0x1D, 0x0F,
	0x05, 0xBF, 0x01, 0xB5, 0x68, 0x82, 0x54, 0xEA, 0xE2, 0xBA, 0xB3, 0xAA,
	0xF7, 0xD5, 0x55, 0xA6, 0x27, 0xF8, 0x9F, 0x02, 0x73, 0x23, 0x80, 0x9B,
	0x5C, 0x68, 0x62, 0x10, 0x40, 0xAB, 0xDF, 0x8B, 0xAD, 0x7D, 0xF3, 0x62,
	0x43, 0x2E, 0xD8, 0xBF, 0x88, 0x6B, 0xCE, 0x9A, 0xF1, 0x5D, 0xE3, 0x0D,
	0x9A, 0x37, 0xFD, 0x3C, 0x34, 0x5A, 0x01, 0xC2, 0x55, 0x6E, 0xB5, 0x33,
	0x86, 0xD3, 0x04, 0x7A, 0xD7, 0x93, 0x4C, 0x6E, 0x3C, 0xBC, 0x60, 0xF7,
	0x3F, 0x6E, 0x34, 0x20, 0xDE, 0x59, 0x0C, 0x0A, 0xF6, 0x36, 0xD4, 0x33,
	0xF3, 0x16, 0x00, 0x95, 0xC2, 0xED, 0x1B, 0x33, 0x0C, 0xC2, 0xCB, 0xEA,
	0xD1, 0x73, 0xDB, 0xF0, 0xC8, 0xDE, 0x8C, 0x48, 0xC3, 0x42, 0x9D, 0x15,
	0x50, 0x96, 0x54, 0x01, 0xDE, 0xDD, 0x04, 0xD4, 0x88, 0xB4, 0x6B, 0x0B,
	0x86, 0x02, 0x60, 0x89, 0x1A, 0x89, 0xFD, 0x21, 0xCA, 0xA4, 0x24, 0xCA,
	0x50, 0xA0, 0xBB, 0xE1, 0x21, 0x82, 0x77, 0x7F, 0x49, 0x38, 0x0A, 0x00,
	0x08, 0x61, 0x10, 0x0E, 0x04, 0x94, 0xE8, 0xC3, 0xA2, 0x4C, 0x0A, 0x71,
	0x96, 0xDC, 0x13, 0x7D, 0x9A, 0x88, 0x3F, 0x01, 0x30, 0x45, 0x84, 0x3D,
	0x81, 0x54, 0x10, 0xF0, 0x61, 0x50, 0xA9, 0x5F, 0x2B, 0xC2, 0x98, 0xB3,
	0x61, 0xC8, 0xA0, 0xA0, 0x12, 0x7D, 0x93, 0x09, 0xCF, 0x8A, 0x10, 0xE3,
	0x26, 0x4C, 0xD8, 0x94, 0xA8, 0x89, 0x35, 0x83, 0xC0, 0x76, 0x6D, 0x78,
	0x9C, 0x08, 0x08, 0x2E, 0x8A, 0x6E, 0xBA, 0x1C, 0x1C, 0x05, 0x00, 0x04,
	0x84, 0xAF, 0x3B, 0x7A, 0xFB, 0xE4, 0x7F, 0x77, 0xFE, 0xF2, 0xA5, 0x5D,
	0x1B, 0xB6, 0x87, 0xA1, 0xAC, 0x44, 0x9B, 0x2F, 0x17, 0x47, 0x8D, 0x42,
	0x8C, 0x35, 0x81, 0x9E, 0x06, 0xDB, 0x9A, 0x6D, 0x0D, 0xC3, 0x19, 0x4A,
	0x74, 0xB1, 0x06, 0x7C, 0x62, 0xB7, 0x7E, 0x91, 0x61, 0xD6, 0x78, 0x71,
	0xA2, 0xB6, 0xF3, 0x33, 0xAB, 0x15, 0x2D, 0x3F, 0x6C, 0xF0, 0xDB, 0xC6,
	0x9B, 0x34, 0x6F, 0xE6, 0x67, 0x02, 0xA6, 0x5A, 0xAD, 0x5B, 0x42, 0x0C,
	0x6B, 0x92, 0x74, 0x67, 0x7F, 0x4D, 0x7B, 0xBF, 0x95, 0x4A, 0xD6, 0x86,
	0x21, 0xB7, 0x48, 0xEC, 0xCD, 0xEC, 0xB9, 0xCC, 0x1D, 0x05, 0x00, 0xD7,
	0x4B, 0x9A, 0xD6, 0x09, 0x6E, 0xB1, 0xF4, 0xFC, 0x96, 0x0A, 0xCB, 0x4A,
	0x7C, 0x05, 0x80, 0x90, 0x25, 0x59, 0xA5, 0x4B, 0x24, 0xA0, 0xA8, 0x4D,
	0x56, 0x2A, 0x98, 0x1E, 0x86, 0xB3, 0x7E, 0x88, 0x4E, 0x4D, 0xA6, 0xD1,
	0x87, 0xE2, 0x2C, 0x61, 0xDC, 0xE2, 0xB8, 0x37, 0x99, 0x9A, 0x69, 0x76,
	0x2D, 0x69, 0xBA, 0x67, 0x25, 0x33, 0xBC, 0x0E, 0x57, 0x96, 0xA3, 0x00,
	0x60, 0x4A, 0xDA, 0xEF, 0x5D, 0x63, 0xB6, 0xB0, 0xA9, 0x9E, 0x75, 0xA1,
	0x57, 0xF5, 0x03, 0xB8, 0xDA, 0xAE, 0xAA, 0x92, 0x85, 0x71, 0xC6, 0x9B,
	0x4A, 0x4D, 0x37, 0xD3, 0xBB, 0x4C, 0xF5, 0xAC, 0x64, 0x86, 0x56, 0xE1,
	0x4A, 0x74, 0x14, 0x00, 0x10, 0xAE, 0x4A, 0xFB, 0x7D, 0x4F, 0x98, 0x29,
	0x6A, 0xEC, 0x2C, 0x6E, 0x91, 0x2E, 0x6C, 0xDC, 0x5D, 0xC9, 0x34, 0x9A,
	0xF9, 0x32, 0x1A, 0x16, 0x90, 0xBF, 0x57, 0xEF, 0x85, 0x3B, 0x3B, 0x9C,
	0xA5, 0x44, 0xD5, 0xF4, 0xDE, 0x78, 0xAD, 0x51, 0x21, 0x13, 0x3D, 0x8B,
	0x1E, 0x15, 0x22, 0x47, 0x2C, 0xAD, 0x04, 0xBE, 0x8D, 0x09, 0xF3, 0x00,
	0xEC, 0x13, 0x61, 0x50, 0xD2, 0xA8, 0xCE, 0xA8, 0x8C, 0xD7, 0xD0, 0xCA,
	0xC8, 0xE1, 0x42, 0x09, 0x41, 0x9F, 0xAA, 0xA1, 0x8E, 0xE5, 0xA3, 0xBB,
	0x07, 0x33, 0xBB, 0x9B, 0x16, 0xA7, 0x28, 0xFD, 0x27, 0x80, 0x6B, 0x9C,
	0x58, 0x65, 0xE2, 0x45, 0x46, 0x65, 0x74, 0x7B, 0x56, 0x75, 0x57, 0x93,
	0x0C, 0x67, 0xA7, 0x30, 0x42, 0x21, 0xE0, 0x98, 0xDF, 0xCB, 0xCB, 0xC7,
	0x6E, 0xB3, 0x1C, 0x8A, 0xB4, 0x9E, 0x02, 0xE1, 0xB0, 0x73, 0xDB, 0x24,
	0x8F, 0xEC, 0xF4, 0xE6, 0x47, 0xD7, 0x59, 0x19, 0x4F, 0x26, 0xE2, 0x54,
	0x84, 0x40, 0x34, 0x26, 0x8A, 0x66, 0x9F, 0xE0, 0x54, 0x7D, 0xB7, 0x6C,
	0x1A, 0x18, 0x73, 0xC4, 0x34, 0xE1, 0xD5, 0x7D, 0x6F, 0xE9, 0x3A, 0x8B,
	0x35, 0x9E, 0x2D, 0x46, 0x84, 0x73, 0x98, 0xB0, 0x45, 0x0D, 0x75, 0x7C,
	0x9D, 0x9D, 0x2E, 0x79, 0xA4, 0xED, 0x00, 0x26, 0x8A, 0x68, 0x43, 0x02,
	0xCD, 0xD2, 0xCF, 0xD7, 0x67, 0xA6, 0x08, 0x11, 0xCE, 0xA1, 0xFD, 0x93,
	0x2B, 0xCE, 0xAD, 0xCF, 0x4E, 0x95, 0x95, 0x86, 0x95, 0x04, 0x2C, 0x16,
	0xD5, 0x0A, 0x83, 0x75, 0x9F, 0x57, 0xDF, 0x59, 0x04, 0x59, 0x94, 0x10,
	0x07, 0x9C, 0xD5, 0x88, 0x96, 0xFE, 0x36, 0x67, 0x6F, 0x72, 0x6C, 0x62,
	0x55, 0x6F, 0xFD, 0x6C, 0x80, 0x37, 0x8B, 0x6C, 0x88, 0x81, 0xA0, 0x5E,
	0xBE, 0x51, 0xCF, 0xAA, 0x10, 0xA8, 0xC5, 0x16, 0x44, 0xB4, 0xBA, 0x3F,
	0xD4, 0xDE, 0x37, 0x36, 0x6D, 0xC6, 0x17, 0x4F, 0x4D, 0xF0, 0x68, 0xB4,
	0x0B, 0xC0, 0x64, 0xB1, 0xAD, 0xB1, 0xEE, 0xF3, 0x1A, 0x39, 0xAB, 0xC8,
	0x4B, 0x1C, 0xFA, 0x34, 0x1E, 0xEA, 0xD8, 0x91, 0x9D, 0xAA, 0x55, 0x9C,
	0x78, 0x1D, 0xC0, 0x1D, 0xC2, 0x5B, 0x03, 0xE9, 0x4E, 0x3F, 0xF4, 0x9D,
	0xC5, 0x98, 0x24, 0x50, 0x8B, 0x06, 0xF0, 0xC7, 0x00, 0xB6, 0x03, 0x38,
	0x6E, 0x54, 0xF8, 0xE2, 0x34, 0x21, 0x8B, 0x80, 0xD2, 0x70, 0x3F, 0x80,
	0xD5, 0x02, 0x75, 0x5D, 0x82, 0xF5, 0x7B, 0xAA, 0xD1, 0x3B, 0xEB, 0xAC,
	0x38, 0x21, 0x14, 0x55, 0xC3, 0x9D, 0x0F, 0xAB, 0xE1, 0xD8, 0xE3, 0x9A,
	0x4F, 0x9A, 0x03, 0xE0, 0x57, 0x9D, 0xD2, 0x39, 0xA7, 0x09, 0xD5, 0xFB,
	0xEA, 0x2A, 0x09, 0xDC, 0x0E, 0xB7, 0xF6, 0xFE, 0x0D, 0x9E, 0xD7, 0xC8,
	0x59, 0xA7, 0x05, 0xC9, 0x18, 0x54, 0x23, 0x1D, 0x1F, 0x8C, 0xFE, 0xD1,
	0x7F, 0x77, 0xFB, 0x5F, 0x52, 0xC6, 0x73, 0x1F, 0xF2, 0x38, 0x2C, 0xDF,
	0x34, 0x21, 0xE3, 0xF7, 0xED, 0x04, 0x30, 0x4D, 0x90, 0xA6, 0xF1, 0xED,
	0x82, 0x4F, 0xE9, 0xE5, 0x1B, 0x0D, 0xC3, 0x93, 0x82, 0x74, 0x4C, 0xBA,
	0xEB, 0xC7, 0x15, 0xBE, 0xB1, 0x09, 0x47, 0xEE, 0x69, 0xFB, 0x3B, 0xB7,
	0xC3, 0xF2, 0x4F, 0x13, 0x18, 0x78, 0x48, 0x90, 0x9E, 0x3C, 0xD0, 0x09,
	0xBD, 0x5C, 0xA3, 0x9E, 0x15, 0x17, 0xA4, 0xA2, 0xF2, 0xF8, 0xF9, 0x73,
	0x2F, 0x65, 0x27, 0x8E, 0x73, 0x18, 0xE3, 0x4C, 0xA1, 0xA6, 0x09, 0xB9,
	0x20, 0xE8, 0x3F, 0xAF, 0xBE, 0xB3, 0x34, 0xE7, 0x6B, 0xAE, 0x4B, 0x42,
	0x68, 0x7D, 0x50, 0x69, 0x18, 0xD7, 0x63, 0xC6, 0x3A, 0x8C, 0x89, 0x0B,
	0x38, 0x4D, 0xC8, 0xA9, 0xF1, 0x90, 0x5E, 0xBE, 0xFE, 0x72, 0x87, 0xF8,
	0xA0, 0x48, 0x31, 0x0C, 0x7E, 0x55, 0xEE, 0x89, 0x3E, 0x97, 0x9D, 0x7E,
	0xE4, 0x9E, 0xB6, 0xBF, 0x31, 0x61, 0x62, 0x6D, 0x22, 0xDC, 0xF9, 0x5E,
	0x76, 0x9E, 0x5B, 0xD3, 0x84, 0x9C, 0xFA, 0x18, 0xBF, 0xEB, 0xE5, 0xEB,
	0x3A, 0xCB, 0xA7, 0xF9, 0x14, 0xB1, 0x72, 0x00, 0x30, 0x36, 0xE4, 0xEA,
	0x61, 0xEA, 0xDC, 0x1D, 0xE3, 0xDE, 0x17, 0x81, 0xEE, 0x65, 0x8B, 0x00,
	0x3C, 0x23, 0x5C, 0x43, 0x1E, 0x98, 0x52, 0xDD, 0x7A, 0xF9, 0x86, 0x9F,
	0x60, 0x59, 0x89, 0x26, 0x00, 0x4C, 0x17, 0x25, 0xE8, 0x52, 0xC3, 0xF4,
	0x62, 0x3C, 0xDC, 0xF1, 0x5A, 0xBE, 0xFC, 0xEA, 0x7D, 0x75, 0x95, 0x69,
	0xBF, 0xEF, 0x00, 0x5C, 0xFC, 0xFA, 0x65, 0x71, 0x44, 0x0D, 0xC7, 0x6E,
	0xD1, 0x2B, 0x60, 0xB8, 0x53, 0x3A, 0x12, 0x15, 0x2C, 0x9E, 0x7C, 0x43,
	0x72, 0x14, 0xB7, 0xA7, 0x09, 0xE3, 0x20, 0x32, 0x7C, 0x4E, 0x43, 0x67,
	0x31, 0xD1, 0x1E, 0x31, 0x6A, 0x72, 0x19, 0xCF, 0x3D, 0x24, 0x0B, 0x33,
	0x4D, 0xC8, 0x92, 0xC2, 0xD8, 0x6D, 0x54, 0xC6, 0x78, 0x26, 0xCC, 0x2D,
	0x92, 0xDC, 0xA3, 0xF6, 0xC3, 0xC5, 0x43, 0x0B, 0x06, 0xBF, 0x36, 0x65,
	0xC2, 0xA4, 0x57, 0x7E, 0x9A, 0xBB, 0x23, 0x25, 0x77, 0xD7, 0x3F, 0x00,
	0xA2, 0x8F, 0x00, 0xA1, 0x4B, 0x2D, 0x23, 0x06, 0xD4, 0x50, 0x2C, 0x60,
	0x14, 0xE8, 0x66, 0x7C, 0x60, 0x31, 0x12, 0x90, 0xDF, 0x2A, 0x4A, 0x55,
	0xCE, 0x26, 0x40, 0xEB, 0x87, 0xCF, 0xFF, 0x77, 0x4C, 0x56, 0xA2, 0x47,
	0x40, 0xF4, 0x39, 0x0A, 0xEB, 0x28, 0x80, 0xB9, 0xD5, 0x4C, 0x44, 0xA0,
	0xA9, 0x43, 0x56, 0xBF, 0x17, 0x5B, 0x01, 0x61, 0x4B, 0x9F, 0x7C, 0x54,
	0xC2, 0x60, 0x3F, 0xC9, 0x25, 0x4E, 0x7B, 0x53, 0xE9, 0x77, 0xCC, 0x14,
	0x34, 0xE5, 0xAC, 0xBE, 0x79, 0xB1, 0x21, 0x02, 0xBD, 0xEB, 0x4C, 0x53,
	0x89, 0xC2, 0xD8, 0x26, 0x3C, 0x30, 0xC4, 0xE7, 0xE5, 0x0D, 0x00, 0x5C,
	0xB9, 0xB9, 0x50, 0x2C, 0x18, 0x18, 0xF2, 0xA6, 0x52, 0x5B, 0xCC, 0x96,
	0x37, 0xED, 0xAC, 0xBE, 0x79, 0xB1, 0x21, 0x22, 0x5A, 0x67, 0x4F, 0x56,
	0x69, 0x42, 0x84, 0xB5, 0x56, 0xAE, 0xAE, 0x58, 0x0A, 0x66, 0x8B, 0xD7,
	0x04, 0xDE, 0xBF, 0x70, 0x73, 0xE1, 0x0A, 0x80, 0xBA, 0xD4, 0x9A, 0x58,
	0x9B, 0x95, 0x1A, 0xD6, 0xC2, 0x24, 0xA9, 0x45, 0xA3, 0x8C, 0xA7, 0x8E,
	0x01, 0x57, 0x6F, 0x5F, 0x15, 0x80, 0x61, 0x46, 0xAA, 0xDE, 0x6A, 0x4C,
	0xBC, 0xE5, 0xD0, 0xEE, 0xF8, 0xFC, 0xB6, 0x41, 0x68, 0xDC, 0x08, 0xD8,
	0x0F, 0xBE, 0x2F, 0x32, 0x1A, 0x40, 0xF5, 0x89, 0xF0, 0xAE, 0x01, 0xAB,
	0x15, 0x6D, 0xC5, 0xC1, 0x27, 0x6A, 0x3B, 0x3F, 0x63, 0xC6, 0x2A, 0x3B,
	0x75, 0x4B, 0x80, 0x35, 0x6A, 0xB8, 0xE3, 0x73, 0x3B, 0x15, 0x6D, 0x5F,
	0x1A, 0x48, 0x44, 0x62, 0xDB, 0x98, 0xB0, 0xC9, 0x6E, 0xFD, 0xA2, 0xC0,
	0xD8, 0xA8, 0x86, 0x63, 0x6F, 0xD9, 0xAD, 0xEE, 0xE8, 0xA2, 0x53, 0xA2,
	0x26, 0xD6, 0x7C, 0x19, 0x39, 0xEC, 0x0D, 0x35, 0x1C, 0x7B, 0xC1, 0x89,
	0x01, 0x21, 0xA7, 0x24, 0x41, 0xA5, 0x61, 0x15, 0x83, 0xDF, 0x86, 0xC0,
	0x2B, 0x79, 0x02, 0x61, 0x62, 0x34, 0xC7, 0x23, 0x31, 0xC7, 0xFF, 0x54,
	0x61, 0x47, 0x4A, 0x72, 0x4F, 0xF4, 0x41, 0x66, 0xB4, 0x97, 0xD2, 0x85,
	0x02, 0x06, 0x86, 0x08, 0xD4, 0x60, 0xF7, 0x1D, 0x95, 0x8D, 0xD0, 0xF3,
	0x37, 0xB9, 0x3B, 0x7A, 0x33, 0x08, 0x1F, 0x00, 0x28, 0x81, 0x50, 0x25,
	0xEA, 0xA2, 0xB4, 0xB4, 0x34, 0x3E, 0xBF, 0x6D, 0x50, 0x98, 0x45, 0x51,
	0x86, 0x2E, 0x72, 0xE9, 0x8E, 0xF4, 0x66, 0x00, 0x37, 0x08, 0xB7, 0x6F,
	0xCC, 0x71, 0x10, 0x5E, 0x51, 0x6B, 0xE4, 0xAD, 0xA2, 0x7F, 0xC2, 0xC0,
	0xB5, 0x5B, 0x5D, 0xB3, 0x7E, 0x88, 0x4E, 0x3D, 0x9F, 0x41, 0x33, 0x31,
	0x56, 0xA2, 0x30, 0x31, 0x13, 0xA7, 0x00, 0xBC, 0x93, 0x24, 0xDF, 0x9B,
	0x47, 0x43, 0x3B, 0x0D, 0xC3, 0x03, 0xEC, 0xE0, 0xFA, 0x15, 0xB8, 0x1B,
	0x7B, 0x1E, 0x9B, 0xE2, 0xD7, 0x92, 0x4F, 0x82, 0xA8, 0x09, 0xEE, 0x84,
	0x5C, 0x0E, 0x80, 0xB9, 0x35, 0x29, 0xF9, 0xB7, 0xB9, 0xE5, 0xA4, 0x51,
	0x0A, 0xFA, 0x8B, 0x21, 0xD3, 0x7B, 0xE3, 0xB5, 0xC4, 0x58, 0x02, 0x60,
	0x21, 0x81, 0x9C, 0xC4, 0x7E, 0xC5, 0xC1, 0xF8, 0x8A, 0x89, 0x76, 0x25,
	0x42, 0x1D, 0x5D, 0x4E, 0xAE, 0xF2, 0x5A, 0xA1, 0x68, 0x97, 0x2B, 0x03,
	0xCA, 0x92, 0x2A, 0x62, 0x5F, 0x84, 0x88, 0x6F, 0x1D, 0x89, 0xB8, 0x93,
	0x02, 0x0C, 0xED, 0x3A, 0x02, 0x5D, 0x8B, 0x91, 0x28, 0xEA, 0x34, 0x83,
	0x4F, 0x12, 0xD1, 0x30, 0x18, 0x09, 0x02, 0x0E, 0x33, 0xD3, 0x41, 0xA6,
	0x54, 0xB7, 0x9D, 0xA5, 0x4A, 0x99, 0x32, 0x65, 0xCA, 0x94, 0x29, 0x53,
	0xA6, 0x4C, 0x81, 0xF9, 0x1F, 0x25, 0x02, 0x69, 0xE7, 0xF6, 0x40, 0xE0,
	0x41, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45, 0x4E, 0x44, 0xAE, 0x42, 0x60,
	0x82
};

static unsigned char warning_[1614] = {
	0x89, 0x50, 0x4E, 0x47, 0x0D, 0x0A, 0x1A, 0x0A, 0x00, 0x00, 0x00, 0x0D,
	0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x4B, 0x00, 0x00, 0x00, 0x4B,
	0x08, 0x06, 0x00, 0x00, 0x00, 0x38, 0x4E, 0x7A, 0xEA, 0x00, 0x00, 0x00,
	0x04, 0x73, 0x42, 0x49, 0x54, 0x08, 0x08, 0x08, 0x08, 0x7C, 0x08, 0x64,
	0x88, 0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x2E,
	0x23, 0x00, 0x00, 0x2E, 0x23, 0x01, 0x78, 0xA5, 0x3F, 0x76, 0x00, 0x00,
	0x00, 0x19, 0x74, 0x45, 0x58, 0x74, 0x53, 0x6F, 0x66, 0x74, 0x77, 0x61,
	0x72, 0x65, 0x00, 0x77, 0x77, 0x77, 0x2E, 0x69, 0x6E, 0x6B, 0x73, 0x63,
	0x61, 0x70, 0x65, 0x2E, 0x6F, 0x72, 0x67, 0x9B, 0xEE, 0x3C, 0x1A, 0x00,
	0x00, 0x05, 0xCB, 0x49, 0x44, 0x41, 0x54, 0x78, 0x9C, 0xED, 0x9B, 0x5D,
	0x6C, 0x14, 0x55, 0x14, 0xC7, 0xFF, 0xE7, 0xCE, 0xB6, 0xD0, 0x99, 0xD9,
	0x36, 0x14, 0x10, 0x35, 0x01, 0x4C, 0x89, 0xA0, 0xA8, 0x18, 0xB0, 0x4A,
	0xC4, 0xE8, 0x03, 0x18, 0x78, 0x20, 0x94, 0xF8, 0x60, 0x63, 0xD5, 0xA0,
	0x2D, 0x5F, 0xD1, 0x44, 0xF1, 0x23, 0x3C, 0x98, 0x18, 0xB3, 0x41, 0x12,
	0x45, 0x13, 0x3E, 0x1E, 0x78, 0x40, 0xD8, 0x2D, 0x9A, 0x82, 0x49, 0xDF,
	0x8C, 0xC4, 0x18, 0xC3, 0x8B, 0xBE, 0x18, 0x93, 0x12, 0x04, 0x89, 0x82,
	0x0A, 0xB1, 0xC6, 0x44, 0x14, 0x4A, 0x80, 0xDD, 0x99, 0x65, 0xE9, 0xCE,
	0x1C, 0x5F, 0x0A, 0x52, 0x9C, 0xD9, 0xCE, 0xCE, 0xDC, 0x3B, 0xB3, 0x92,
	0xF9, 0xBD, 0xCD, 0xCE, 0xBD, 0xFF, 0x73, 0xE6, 0xBF, 0x67, 0x76, 0xEF,
	0x9D, 0xB3, 0x0B, 0xA4, 0xA4, 0xA4, 0xA4, 0xFC, 0x7F, 0xA0, 0xA4, 0x13,
	0xB8, 0x91, 0x91, 0x81, 0xF6, 0xD6, 0x49, 0xE5, 0xCA, 0x5D, 0xC8, 0x60,
	0x06, 0x80, 0xB3, 0x95, 0xA6, 0x49, 0xC3, 0x53, 0x9F, 0xBF, 0x70, 0x39,
	0xE9, 0xBC, 0xAE, 0x91, 0xB8, 0x59, 0x3C, 0x88, 0x66, 0xBB, 0xA4, 0xF7,
	0x02, 0xB4, 0x01, 0xC0, 0x22, 0x8F, 0x21, 0x47, 0x00, 0xFE, 0x48, 0x37,
	0xED, 0xFD, 0xD4, 0x8D, 0xAB, 0x71, 0xE7, 0x77, 0x23, 0x89, 0x9A, 0x55,
	0xDA, 0x6B, 0x2C, 0x20, 0x0D, 0x9F, 0x02, 0x98, 0x1F, 0x60, 0xF8, 0x09,
	0xD7, 0xA5, 0x9E, 0xEC, 0xBA, 0xD2, 0x09, 0xD5, 0x79, 0xF9, 0x91, 0x98,
	0x59, 0x76, 0xA1, 0x65, 0x09, 0x20, 0xBE, 0x64, 0x20, 0x5B, 0xC7, 0xB4,
	0xCB, 0x04, 0x77, 0xB9, 0xDE, 0x57, 0xFE, 0x4E, 0x59, 0x62, 0x35, 0x48,
	0xC4, 0x2C, 0x2B, 0xAF, 0xDF, 0x09, 0xA2, 0xE3, 0x00, 0xA6, 0x86, 0x98,
	0x7E, 0x8E, 0x09, 0x0B, 0xCC, 0x5E, 0xEB, 0xAC, 0xEC, 0xBC, 0x26, 0x42,
	0xC4, 0x1D, 0x70, 0x2C, 0xEC, 0x6E, 0x84, 0x33, 0x0A, 0x00, 0xA6, 0x13,
	0xF3, 0x6E, 0x99, 0xD9, 0x04, 0x25, 0xF6, 0xCA, 0x2A, 0xED, 0x37, 0x96,
	0x91, 0x8B, 0xC3, 0x51, 0x75, 0x98, 0xB1, 0xC2, 0x5C, 0x6B, 0x7D, 0x25,
	0x23, 0xA7, 0xA0, 0xC4, 0x5A, 0x59, 0x3C, 0x08, 0x8D, 0x5C, 0xEC, 0x90,
	0xA1, 0x45, 0x84, 0x1D, 0x9C, 0x43, 0x46, 0x86, 0x56, 0x50, 0x62, 0x35,
	0xAB, 0x6C, 0x99, 0x2F, 0x03, 0x78, 0x40, 0x92, 0xDC, 0x7C, 0x7B, 0xB6,
	0xB9, 0x41, 0x92, 0x56, 0x20, 0x62, 0xBB, 0x0D, 0x2F, 0x1E, 0x68, 0x9B,
	0xD2, 0x54, 0xA9, 0xFE, 0x82, 0xF0, 0x9F, 0x55, 0x5E, 0x5C, 0x70, 0x34,
	0x31, 0xB7, 0xF5, 0x85, 0xE2, 0x88, 0x44, 0x4D, 0x5F, 0x62, 0xAB, 0xAC,
	0xA6, 0xAB, 0xA3, 0xEF, 0x42, 0xAE, 0x51, 0x00, 0xD0, 0x2E, 0x1C, 0xF7,
	0x1D, 0xC9, 0x9A, 0xBE, 0xC4, 0x52, 0x59, 0xC5, 0x82, 0x39, 0x5F, 0x80,
	0xBF, 0x07, 0xD0, 0xA4, 0x40, 0xBE, 0xEA, 0xBA, 0xB4, 0x30, 0x8E, 0xC5,
	0x6A, 0x2C, 0x95, 0x25, 0xC0, 0xDB, 0xA1, 0xC6, 0x28, 0x00, 0xC8, 0x68,
	0xC4, 0x3B, 0x15, 0x69, 0x8F, 0x43, 0xB9, 0x59, 0x56, 0x41, 0x5F, 0x0D,
	0x60, 0x85, 0xCA, 0x18, 0x4C, 0x58, 0x66, 0xF5, 0xEB, 0xAB, 0x54, 0xC6,
	0x00, 0x14, 0xDF, 0x86, 0x3C, 0x88, 0x66, 0xAB, 0x68, 0xFC, 0x40, 0x84,
	0xB9, 0x2A, 0xE3, 0x8C, 0x45, 0x3B, 0xAD, 0x97, 0xEC, 0xFB, 0xE8, 0x55,
	0x54, 0x54, 0x45, 0x50, 0x5A, 0x59, 0x96, 0x65, 0xBC, 0x1E, 0x8F, 0x51,
	0x00, 0x40, 0x73, 0x2C, 0xD3, 0x78, 0x45, 0x69, 0x04, 0x55, 0xC2, 0xC5,
	0x4F, 0xCC, 0xDB, 0x44, 0x95, 0x7F, 0x06, 0xD0, 0xA6, 0x2A, 0xC6, 0xCD,
	0x10, 0x50, 0xE4, 0x51, 0x9E, 0x67, 0x6C, 0xB4, 0xFF, 0x54, 0xA1, 0xAF,
	0xAC, 0xB2, 0x44, 0x95, 0xDF, 0x43, 0x8C, 0x46, 0x01, 0x00, 0x03, 0x59,
	0x34, 0x89, 0x2D, 0xAA, 0xF4, 0x95, 0x54, 0x96, 0xB5, 0x4F, 0x5F, 0x08,
	0x41, 0x43, 0x48, 0x66, 0xA3, 0xEE, 0x82, 0x79, 0xB1, 0xB1, 0xD6, 0x1E,
	0x92, 0x2D, 0xAC, 0xE4, 0x62, 0x58, 0xA3, 0x9D, 0xAA, 0xB4, 0x03, 0x20,
	0x40, 0xD8, 0xC5, 0x2C, 0xBF, 0x10, 0xA4, 0x5F, 0x90, 0x5D, 0x30, 0x7A,
	0x88, 0xF1, 0x84, 0x6C, 0xDD, 0xFA, 0xA0, 0x25, 0xE5, 0x7E, 0xB3, 0x5B,
	0xBA, 0xAA, 0x4C, 0x31, 0xDE, 0x03, 0xDD, 0x6E, 0x32, 0x7E, 0x02, 0x30,
	0x2B, 0x82, 0x4C, 0x19, 0xC0, 0x19, 0x00, 0x1D, 0x00, 0x5A, 0x22, 0xE8,
	0x0C, 0xEB, 0xA6, 0x75, 0x2F, 0x75, 0xA3, 0x1C, 0x41, 0x63, 0x1C, 0x52,
	0x2B, 0xCB, 0x6E, 0xD6, 0x37, 0x23, 0x9A, 0x51, 0x47, 0x99, 0xD0, 0x61,
	0xF4, 0x59, 0xF7, 0x63, 0x94, 0xE7, 0x00, 0x38, 0x16, 0x41, 0x6B, 0xB6,
	0x5D, 0xD2, 0x37, 0x47, 0x98, 0xFF, 0x1F, 0xA4, 0x55, 0x96, 0x5D, 0x68,
	0x99, 0xC9, 0x10, 0x27, 0x01, 0xE8, 0xE1, 0xB3, 0xE1, 0x95, 0x46, 0xAF,
	0xFD, 0xC5, 0xB5, 0x43, 0x6B, 0x9F, 0xBE, 0x12, 0x82, 0x0E, 0x45, 0x48,
	0xCB, 0x22, 0xC7, 0xBD, 0x47, 0x5F, 0x5F, 0xFE, 0x23, 0x82, 0xC6, 0x75,
	0xA4, 0x55, 0x96, 0x0B, 0xB1, 0x0D, 0x51, 0x8C, 0x02, 0xE0, 0x40, 0x3B,
	0x33, 0xEE, 0x58, 0xD3, 0x4E, 0x47, 0x4A, 0x0A, 0x30, 0xA0, 0x89, 0x6D,
	0x11, 0x35, 0xAE, 0x23, 0xC5, 0x2C, 0x3B, 0xDF, 0xF2, 0x28, 0x01, 0xCF,
	0xC8, 0xD0, 0x92, 0x0D, 0x03, 0x3D, 0xE5, 0xFE, 0xEC, 0xE3, 0x32, 0xB4,
	0x22, 0x9B, 0xC5, 0x39, 0x08, 0x26, 0xB1, 0x0B, 0x0D, 0xD0, 0xB0, 0xF5,
	0x81, 0x5C, 0x76, 0x77, 0x72, 0x2E, 0xFA, 0xB5, 0x46, 0x16, 0xB0, 0x67,
	0xEB, 0x2F, 0x02, 0x78, 0x38, 0xAA, 0x8E, 0x62, 0x16, 0x59, 0x33, 0x8D,
	0x35, 0x51, 0x45, 0x22, 0x99, 0x75, 0x2E, 0x3F, 0x2D, 0x0B, 0xD0, 0xD6,
	0xA8, 0x49, 0xC4, 0x01, 0x11, 0xDE, 0x1F, 0x19, 0x68, 0x6F, 0x8D, 0xA2,
	0x11, 0xC9, 0x2C, 0x83, 0xCA, 0x6F, 0x83, 0x71, 0x47, 0x14, 0x8D, 0x18,
	0x99, 0xD1, 0x32, 0x5A, 0x79, 0x2B, 0x8A, 0x40, 0x68, 0xB3, 0xAE, 0x14,
	0x26, 0x77, 0x30, 0xB0, 0x29, 0x4A, 0xF0, 0xB8, 0x61, 0xC6, 0x1B, 0x57,
	0xFA, 0x5B, 0xEF, 0x0E, 0x3B, 0x3F, 0xB4, 0x59, 0x0E, 0x32, 0xDB, 0x01,
	0x4C, 0x0A, 0x3B, 0x3F, 0x21, 0x9A, 0x1D, 0x76, 0x43, 0x2F, 0x25, 0x42,
	0x99, 0x55, 0x2C, 0x98, 0x4B, 0x01, 0x5E, 0x1D, 0x36, 0x68, 0xB2, 0xF0,
	0x53, 0xA5, 0xBC, 0xB1, 0x3C, 0xCC, 0xCC, 0xBA, 0xCD, 0xE2, 0x41, 0x68,
	0x02, 0xF1, 0x34, 0x08, 0x54, 0x11, 0xB6, 0x9B, 0x5D, 0xB7, 0x59, 0xE5,
	0x92, 0xF9, 0x12, 0xE4, 0x75, 0x95, 0x93, 0x22, 0x54, 0x37, 0xBB, 0x2E,
	0xB3, 0x2E, 0xED, 0x6B, 0x6D, 0x67, 0x70, 0xAE, 0xDE, 0x20, 0x0D, 0x09,
	0xF3, 0x96, 0x8B, 0x07, 0xDA, 0xA6, 0xD4, 0x33, 0xA5, 0x2E, 0xB3, 0x32,
	0x54, 0xCD, 0x41, 0x7E, 0x57, 0xF9, 0x3A, 0x1A, 0x9C, 0x8E, 0x5A, 0xC7,
	0x92, 0x99, 0x9A, 0xA9, 0x54, 0x73, 0xF5, 0x4C, 0x08, 0xBC, 0x45, 0x19,
	0xEB, 0x2A, 0x1F, 0x03, 0x94, 0xFE, 0x72, 0x65, 0x88, 0x09, 0xAB, 0xCC,
	0x5E, 0xEB, 0x6C, 0xA9, 0xDF, 0xB8, 0x9D, 0x18, 0x87, 0x00, 0x3C, 0xA4,
	0x30, 0x5E, 0xD5, 0x05, 0x3D, 0x98, 0xED, 0x2B, 0xFD, 0x18, 0x64, 0x70,
	0xE0, 0xCA, 0x12, 0xC4, 0x3B, 0xA0, 0xD6, 0x28, 0x00, 0xE8, 0x24, 0xC6,
	0x6F, 0xA5, 0xBC, 0x71, 0x8A, 0x18, 0xC3, 0x50, 0x6B, 0x14, 0x00, 0x64,
	0xC6, 0xBA, 0xE5, 0x81, 0x08, 0x54, 0x59, 0x56, 0x5E, 0xEF, 0x02, 0xD1,
	0x67, 0xE1, 0x73, 0x6A, 0x70, 0x88, 0xBB, 0x8C, 0x5E, 0xFB, 0xF3, 0x09,
	0x87, 0x4D, 0x34, 0x20, 0xDE, 0xAE, 0x72, 0x52, 0x04, 0xEB, 0x66, 0x4F,
	0x78, 0x1B, 0x5A, 0x45, 0x63, 0xD3, 0xAD, 0x6D, 0x14, 0x10, 0xB4, 0x9B,
	0x5D, 0xB3, 0xB2, 0x4A, 0x7B, 0x8D, 0x19, 0xA4, 0xE1, 0x14, 0x62, 0x6E,
	0x96, 0x02, 0x38, 0x0F, 0x60, 0x5A, 0xCC, 0x31, 0x2F, 0xB1, 0x83, 0x79,
	0xE6, 0x7A, 0xEB, 0x2F, 0xBF, 0x01, 0x35, 0x2B, 0x8B, 0x34, 0xDA, 0x8A,
	0x78, 0x8D, 0x3A, 0x21, 0x1C, 0xA7, 0xC3, 0xE8, 0xB3, 0xA6, 0x6B, 0x70,
	0xE6, 0x00, 0x88, 0xF3, 0x0F, 0x02, 0x6D, 0x63, 0xD7, 0xEB, 0x8B, 0x6F,
	0x65, 0x25, 0xD2, 0x55, 0xBE, 0xE9, 0x83, 0x36, 0x81, 0x2F, 0x16, 0x17,
	0x2E, 0x77, 0x1A, 0xEB, 0xEC, 0xA3, 0x5E, 0x27, 0x7D, 0x8D, 0x20, 0xA2,
	0x0F, 0x6B, 0x9D, 0x57, 0x01, 0x39, 0x3C, 0xEE, 0x16, 0x20, 0xE6, 0xB8,
	0xFF, 0x18, 0x20, 0x20, 0xE8, 0x03, 0xFF, 0x93, 0x1E, 0x58, 0x79, 0xBD,
	0x93, 0x09, 0xCB, 0xD4, 0xE5, 0xE4, 0x8D, 0xAB, 0x89, 0x9E, 0x71, 0xC7,
	0x42, 0x3C, 0x1B, 0x77, 0x0E, 0x00, 0x9E, 0xB4, 0xF6, 0xEB, 0x9E, 0xEB,
	0x3B, 0xCF, 0x45, 0x26, 0x09, 0xEA, 0x66, 0x56, 0x9B, 0x91, 0x67, 0x5C,
	0xC6, 0x6B, 0x56, 0xC1, 0xEC, 0x00, 0xBB, 0x43, 0x20, 0xD1, 0x09, 0x70,
	0x57, 0xFC, 0x59, 0x00, 0xE4, 0xD2, 0xD3, 0x00, 0x8E, 0xDC, 0xFC, 0xBA,
	0xA7, 0x59, 0x2E, 0xE3, 0x91, 0xE4, 0x5A, 0x35, 0xDC, 0x05, 0xA2, 0x2E,
	0x20, 0x81, 0x77, 0xEB, 0x5F, 0x3C, 0x1B, 0x30, 0xDE, 0x9F, 0x49, 0xFF,
	0x9F, 0xE7, 0xEA, 0x4A, 0x60, 0x78, 0x5F, 0xBF, 0xA7, 0x59, 0x44, 0x70,
	0xD4, 0xA6, 0xD3, 0xF0, 0x78, 0x5E, 0xBF, 0xDF, 0xB7, 0xDD, 0xEF, 0x0A,
	0x13, 0x69, 0x7C, 0x08, 0xC3, 0x5E, 0x2F, 0x7B, 0x9A, 0xC5, 0xC0, 0xD7,
	0x6A, 0xB3, 0x69, 0x6C, 0x98, 0xF1, 0x8D, 0xD7, 0xEB, 0xDE, 0x66, 0x65,
	0x28, 0x0F, 0xA8, 0xFB, 0x89, 0x74, 0x83, 0x53, 0x81, 0x83, 0x8F, 0xBD,
	0x4E, 0x78, 0x9A, 0x95, 0x5D, 0x53, 0xFA, 0x1B, 0xC0, 0x41, 0xA5, 0x29,
	0x35, 0x2A, 0x84, 0x01, 0xBF, 0xFD, 0xA1, 0xEF, 0x0A, 0xDD, 0xE6, 0x96,
	0x4D, 0x04, 0x9C, 0x52, 0x97, 0x55, 0x23, 0xC2, 0xBF, 0x56, 0xAE, 0x36,
	0xBF, 0xE9, 0x77, 0xB6, 0xE6, 0x72, 0x6A, 0x6C, 0x7F, 0x78, 0x18, 0x40,
	0xBB, 0xF4, 0xBC, 0x1A, 0x8F, 0x11, 0x76, 0xB0, 0xD4, 0x5C, 0x6F, 0x1D,
	0xF7, 0x1B, 0x50, 0x73, 0xEF, 0x67, 0xAC, 0xB3, 0x8F, 0x3A, 0x24, 0x1E,
	0x03, 0xE0, 0xB9, 0xB1, 0xBC, 0x85, 0x38, 0xA2, 0x69, 0x62, 0x49, 0x2D,
	0xA3, 0x80, 0x80, 0x8F, 0x95, 0x39, 0x87, 0x8C, 0x35, 0xCB, 0x78, 0x8E,
	0x40, 0x1B, 0x01, 0x5E, 0x8C, 0xE4, 0x7E, 0xB6, 0x2D, 0x13, 0x17, 0xCC,
	0xDF, 0xB2, 0xA0, 0x3D, 0x86, 0x61, 0x1D, 0xA4, 0xEE, 0x89, 0xD7, 0x96,
	0x75, 0xEF, 0x6A, 0x46, 0x06, 0xDA, 0x5B, 0x27, 0x8F, 0x5E, 0x99, 0xC7,
	0x4C, 0x71, 0x3F, 0x10, 0x94, 0x06, 0x11, 0x5F, 0xB2, 0x5D, 0xFD, 0xE4,
	0xF4, 0xB5, 0xE7, 0x8B, 0x49, 0xE7, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
	0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0xD2, 0x00, 0xFC, 0x03, 0xB7, 0x7B,
	0xAE, 0xE9, 0x02, 0x76, 0x25, 0xD9, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45,
	0x4E, 0x44, 0xAE, 0x42, 0x60, 0x82
};

