#pragma once 

#define RAD2DEG( x ) ( (float)(x) * (float)(180.f / M_PI) )
#define DEG2RAD( x ) ( (float)(x) * (float)(M_PI / 180.f) )

namespace math {

	struct vector2 {
	public:
		float x, y;

		vector2( ) : x( 0 ), y( 0 ) { }
		vector2( float x, float y ) : x( x ), y( y ) { }

		vector2 operator		+ ( const vector2& other ) const { return { x + other.x, y + other.y }; }
		vector2 operator		- ( const vector2& other ) const { return { x - other.x, y - other.y }; }
		vector2 operator		* ( float offset ) const { return { x * offset, y * offset }; }
		vector2 operator		/ ( float offset ) const { return { x / offset, y / offset }; }

		vector2& operator		*= ( const float other ) { x *= other; y *= other; return *this; }
		vector2& operator		/= ( const float other ) { x /= other; y /= other; return *this; }

		vector2& operator		=  ( const vector2& other ) { x = other.x; y = other.y; return *this; }
		vector2& operator		+= ( const vector2& other ) { x += other.x; y += other.y; return *this; }
		vector2& operator		-= ( const vector2& other ) { x -= other.x; y -= other.y; return *this; }
		vector2& operator		*= ( const vector2& other ) { x *= other.x; y *= other.y; return *this; }
		vector2& operator		/= ( const vector2& other ) { x /= other.x; y /= other.y; return *this; }

		operator bool( ) { return bool( x || y ); }
		friend bool operator	==( const vector2& a, const vector2& b ) { return a.x == b.x && a.y == b.y; }
		friend bool operator	!=( const vector2& a, const vector2& b ) { return !( a == b ); }

		inline uint32_t to_bits( float variable ) const
		{
			std::uint32_t bits;
			std::memcpy( &bits, &variable, sizeof( variable ) );
			return bits;
		}

		inline bool infinite( ) const { return ( to_bits( x ) & 0x7FFFFFFF ) == 0x7F800000 || ( to_bits( y ) & 0x7FFFFFFF ) == 0x7F800000; }
		inline bool nan( ) const { return ( to_bits( x ) & 0x7FFFFFFF ) > 0x7F800000 || ( to_bits( y ) & 0x7FFFFFFF ) > 0x7F800000; }

		inline bool empty( ) const
		{
			constexpr float epsilon = 1e-4f;
			return fabs( x ) < epsilon && fabs( y ) < epsilon;
		}

		inline bool valid( ) const { return !empty( ) && !infinite( ) && !nan( ); }

		inline float dot( const vector2& other ) const { return x * other.x + y * other.y; }
		inline float distance( const vector2& other ) const { return sqrtf( powf( other.x - x, 2.0 ) + powf( other.y - y, 2.0 ) ); }
		inline float magnitude( ) const { return sqrt( ( x * x ) + ( y * y ) ); }

		inline vector2 normalize( )
		{
			x = std::clamp( x, -89.0f, 89.0f );

			while ( y > 180.0f ) y -= 360.0f;
			while ( y < -180.0f ) y += 360.0f;

			return vector2( x, y );
		}

		inline vector2 normalized( )
		{
			float len = magnitude( );
			return vector2( x / len, y / len );
		}
	};

	struct vector3 {
	public:
		float x, y, z;

		vector3( ) : x( 0 ), y( 0 ), z( 0 ) { }
		vector3( float x, float y, float z ) : x( x ), y( y ), z( z ) { }

		vector3( float a ) : x( a ), y( a ), z( a ) { }

		vector3 operator		+ ( const vector3& other ) const { return { x + other.x, y + other.y, z + other.z }; }
		vector3 operator		- ( const vector3& other ) const { return { x - other.x, y - other.y, z - other.z }; }
		vector3 operator		* ( float offset ) const { return { x * offset, y * offset, z * offset }; }
		vector3 operator		/ ( float offset ) const { return { x / offset, y / offset, z / offset }; }

		vector3& operator		*= ( const float other ) { x *= other; y *= other; z *= other; return *this; }
		vector3& operator		/= ( const float other ) { x /= other; y /= other; z /= other; return *this; }

		vector3& operator		=  ( const vector3& other ) { x = other.x; y = other.y; z = other.z; return *this; }
		vector3& operator		+= ( const vector3& other ) { x += other.x; y += other.y; z += other.z; return *this; }
		vector3& operator		-= ( const vector3& other ) { x -= other.x; y -= other.y; z -= other.z; return *this; }
		vector3& operator		*= ( const vector3& other ) { x *= other.x; y *= other.y; z *= other.z; return *this; }
		vector3& operator		/= ( const vector3& other ) { x /= other.x; y /= other.y; z /= other.z; return *this; }

		float& operator			[] ( int index ) { return *( &x + index ); }

		operator bool( ) { return bool( x || y || z ); }
		friend bool operator	== ( const vector3& a, const vector3& b ) { return a.x == b.x && a.y == b.y && a.z == b.z; }
		friend bool operator	!= ( const vector3& a, const vector3& b ) { return !( a == b ); }

		inline uint32_t to_bits( float variable ) const
		{
			std::uint32_t bits;
			std::memcpy( &bits, &variable, sizeof( variable ) );
			return bits;
		}

		inline bool infinite( ) const { return ( to_bits( x ) & 0x7FFFFFFF ) == 0x7F800000 || ( to_bits( y ) & 0x7FFFFFFF ) == 0x7F800000 || ( to_bits( z ) & 0x7FFFFFFF ) == 0x7F800000; }
		inline bool nan( ) const { return ( to_bits( x ) & 0x7FFFFFFF ) > 0x7F800000 || ( to_bits( y ) & 0x7FFFFFFF ) > 0x7F800000 || ( to_bits( z ) & 0x7FFFFFFF ) > 0x7F800000; }

		inline bool empty( ) const
		{
			constexpr float epsilon = 1e-4f;
			return fabs( x ) < epsilon && fabs( y ) < epsilon && fabs( z ) < epsilon;
		}

		inline bool valid( ) const { return !empty( ) && !infinite( ) && !nan( ); }

		inline float dot( const vector3& other ) const { return x * other.x + y * other.y + z * other.z; }
		inline float cross_scalar( const vector3& other, double angle ) const { return magnitude( ) * other.magnitude( ) * std::cos( angle ); }

		inline vector3 cross_product( const vector3& other ) const
		{ 
			vector3 out;

			out.x = y * other.z - z * other.y;
			out.y = z * other.x - x * other.z;
			out.z = x * other.y - y * other.x;

			return out;
		}

		inline float distance( const vector3& other ) const { return sqrtf( powf( other.x - x, 2.0 ) + powf( other.y - y, 2.0 ) + powf( other.z - z, 2.0 ) ); }
		inline float magnitude( ) const { return sqrt( ( x * x ) + ( y * y ) + ( z * z ) ); }

		inline vector3 normalize( )
		{
			x = std::clamp( x, -89.0f, 89.0f );

			while ( y > 180.0f ) y -= 360.0f;
			while ( y < -180.0f ) y += 360.0f;

			return vector3( x, y, z );
		}

		inline vector3 normalized( )
		{
			float len = magnitude( );
			return vector3( x / len, y / len, z / len );
		}

		static inline vector3 zero( ) { return vector3( 0.00f, 0.00f, 0.00f ); }
		static inline vector3 get_right( ) { return vector3( 1.00f, 0.00f, 0.00f ); }
		static inline vector3 get_left( ) { return vector3( -1.00f, 0.00f, 0.00f ); }
		static inline vector3 get_forward( ) { return vector3( 0.00f, 0.00f, 1.00f ); }
		static inline vector3 get_backward( ) { return vector3( 0.00f, 0.00f, -1.00f ); }
		static inline vector3 get_up( ) { return vector3( 0.00f, 1.00f, 0.00f ); }
		static inline vector3 get_down( ) { return vector3( 0.00f, -1.00f, 0.00f ); }
	};

	struct quaternion {
	public:
		float x, y, z, w;

		quaternion( ) : x( 0 ), y( 0 ), z( 0 ), w( 0 ) { }
		quaternion( float x, float y, float z, float w ) : x( x ), y( y ), z( z ), w( w ) { }

		quaternion operator		+ ( const quaternion& other ) const { return { x + other.x, y + other.y, z + other.z, w + other.w }; }
		quaternion operator		- ( const quaternion& other ) const { return { x - other.x, y - other.y, z - other.z, w - other.w }; }
		quaternion operator		* ( float offset ) const { return { x * offset, y * offset, z * offset, w * offset }; }
		quaternion operator		/ ( float offset ) const { return { x / offset, y / offset, z / offset, w / offset }; }

		quaternion& operator		*= ( const float other ) { x *= other; y *= other; z *= other; w *= other; return *this; }
		quaternion& operator		/= ( const float other ) { x /= other; y /= other; z /= other; w /= other; return *this; }

		quaternion& operator		=  ( const quaternion& other ) { x = other.x; y = other.y; z = other.z; w = other.w; return *this; }
		quaternion& operator		+= ( const quaternion& other ) { x += other.x; y += other.y; z += other.z; w += other.w; return *this; }
		quaternion& operator		-= ( const quaternion& other ) { x -= other.x; y -= other.y; z -= other.z; w -= other.w; return *this; }
		quaternion& operator		*= ( const quaternion& other ) { x *= other.x; y *= other.y; z *= other.z; w *= other.w; return *this; }
		quaternion& operator		/= ( const quaternion& other ) { x /= other.x; y /= other.y; z /= other.z; w /= other.w; return *this; }

		operator bool( ) { return bool( x || y || z || w ); }
		friend bool operator	== ( const quaternion& a, const quaternion& b ) { return a.x == b.x && a.y == b.y && a.z == b.z && a.w == b.w; }
		friend bool operator	!= ( const quaternion& a, const quaternion& b ) { return !( a == b ); }

		inline uint32_t to_bits( float variable ) const
		{
			std::uint32_t bits;
			std::memcpy( &bits, &variable, sizeof( variable ) );
			return bits;
		}

		inline bool infinite( ) const { return ( to_bits( x ) & 0x7FFFFFFF ) == 0x7F800000 || ( to_bits( y ) & 0x7FFFFFFF ) == 0x7F800000 || ( to_bits( z ) & 0x7FFFFFFF ) == 0x7F800000 || ( to_bits( w ) & 0x7FFFFFFF ) == 0x7F800000; }
		inline bool nan( ) const { return ( to_bits( x ) & 0x7FFFFFFF ) > 0x7F800000 || ( to_bits( y ) & 0x7FFFFFFF ) > 0x7F800000 || ( to_bits( z ) & 0x7FFFFFFF ) > 0x7F800000 || ( to_bits( w ) & 0x7FFFFFFF ) > 0x7F800000; }

		inline bool empty( ) const
		{
			constexpr float epsilon = 1e-4f;
			return fabs( x ) < epsilon && fabs( y ) < epsilon && fabs( z ) < epsilon && fabs( w ) < epsilon;
		}

		inline bool valid( ) const { return !empty( ) && !infinite( ) && !nan( ); }

		inline float dot( const quaternion& vector ) const { return x * vector.x + y * vector.y + z * vector.z + w * vector.w; }
		inline float distance( quaternion v ) const { return float( sqrtf( powf( v.x - x, 2.0 ) + powf( v.y - y, 2.0 ) + powf( v.z - z, 2.0 ) + powf( v.w - w, 2.0 ) ) ); }
		inline float magnitude( ) const { return sqrt( ( x * x ) + ( y * y ) + ( z * z ) + ( w * w ) ); }

		inline vector3 multiply( const vector3& point ) const
		{
			float num = x * 2.00f;
			float num2 = y * 2.00f;
			float num3 = z * 2.00f;
			float num4 = x * num;
			float num5 = y * num2;
			float num6 = z * num3;
			float num7 = x * num2;
			float num8 = x * num3;
			float num9 = y * num3;
			float num10 = w * num;
			float num11 = w * num2;
			float num12 = w * num3;

			vector3 result;

			result.x = ( 1.00f - ( num5 + num6 ) ) * point.x + ( num7 - num12 ) * point.y + ( num8 + num11 ) * point.z;
			result.y = ( num7 + num12 ) * point.x + ( 1.00f - ( num4 + num6 ) ) * point.y + ( num9 - num10 ) * point.z;
			result.z = ( num8 - num11 ) * point.x + ( num9 + num10 ) * point.y + ( 1.00f - ( num4 + num5 ) ) * point.z;

			return result;
		}

		inline vector3 euler_angles( ) const 
		{
			vector3 angles;

			float sinr_cosp = 2.0f * ( w * x + y * z );
			float cosr_cosp = 1.0f - 2.0f * ( x * x + y * y );
			angles.x = std::atan2( sinr_cosp, cosr_cosp );

			float sinp = 2.0f * ( w * y - z * x );
			if ( std::abs( sinp ) >= 1 )
				angles.y = std::copysign( M_PI / 2.0f, sinp ); 
			else
				angles.y = std::asin( sinp );

			float siny_cosp = 2.0f * ( w * z + x * y );
			float cosy_cosp = 1.0f - 2.0f * ( y * y + z * z );
			angles.z = std::atan2( siny_cosp, cosy_cosp );

			return angles * 57.29578f;
		}

		inline vector3 rotate( const vector3& v ) const
		{
			const float vx = 2.0f * v.x;
			const float vy = 2.0f * v.y;
			const float vz = 2.0f * v.z;
			const float w2 = w * w - 0.5f;
			const float dot2 = ( x * vx + y * vy + z * vz );

			vector3 result( ( vx * w2 + ( y * vz - z * vy ) * w + x * dot2 ),
				( vy * w2 + ( z * vx - x * vz ) * w + y * dot2 ),
				( vz * w2 + ( x * vy - y * vx ) * w + z * dot2 ) );

			if ( !result.valid( ) )
				return vector3( );

			return result;
		}

		inline float yaw( ) const 
		{
			return std::atan2( 2.00f * ( w * y + x * z ), 1.00f - 2.00f * ( y * y + z * z ) ) * 57.29578f;
		}
	};

	template<typename T>
	inline T lerp( const T& a, const T& b, float t ) {
		return a + ( b - a ) * t;
	}

	inline quaternion to_quaternion( const vector2& euler )
	{
		float heading = DEG2RAD( euler.x );
		float attitude = DEG2RAD( euler.y );
		float bank = DEG2RAD( 0 );

		float c1 = std::cos( heading / 2 );
		float s1 = std::sin( heading / 2 );
		float c2 = std::cos( attitude / 2 );
		float s2 = std::sin( attitude / 2 );
		float c3 = std::cos( bank / 2 );
		float s3 = std::sin( bank / 2 );

		float c1c2 = c1 * c2;
		float s1s2 = s1 * s2;

		quaternion quat = quaternion(
			c1c2 * s3 + s1s2 * c3,
			s1 * c2 * c3 + c1 * s2 * s3, c1 * s2 * c3 - s1 * c2 * s3,
			c1c2 * c3 - s1s2 * s3
		);

		return quaternion( quat.y, quat.z, ( quat.x * -1 ), quat.w );
	}

	inline float normalize_yaw( float yaw )
	{
		while ( yaw > 180.0f ) yaw -= 360.0f;
		while ( yaw < -180.0f ) yaw += 360.0f;

		return yaw;
	}

	inline vector3 get_right_vector( const quaternion& q )
	{
		return
		{
			1 - 2 * ( q.y * q.y + q.z * q.z ),
			2 * ( q.x * q.y + q.w * q.z ),
			2 * ( q.x * q.z - q.w * q.y )
		};
	}

	inline vector3 get_up_vector( const quaternion& q )
	{
		return
		{
			2 * ( q.x * q.y - q.w * q.z ),
			1 - 2 * ( q.x * q.x + q.z * q.z ),
			2 * ( q.y * q.z + q.w * q.x )
		};
	}

	inline std::vector<std::pair<vector3, vector2>> generate_sphere_samples( float radius, int num_lat, int num_long, quaternion* player_quat )
	{
		std::vector<std::pair<vector3, vector2>> points;

		const auto right = get_right_vector( *player_quat );
		const auto up = get_up_vector( *player_quat );

		for ( int i = 0; i <= num_lat; ++i ) {

			float theta = i * M_PI / num_lat;

			for ( int j = 0; j <= num_long; ++j ) {

				float phi = j * 2.00f * M_PI / num_long;

				float x = radius * std::sin( theta ) * std::cos( phi );
				float y = radius * std::sin( theta ) * std::sin( phi );
				float z = radius * std::cos( theta );

				const auto rotated = player_quat->multiply( math::vector3( x, y, z ) );

				float x_offset = rotated.x * right.x + rotated.y * right.y + rotated.z * right.z;
				float y_offset = rotated.x * up.x + rotated.y * up.y + rotated.z * up.z;

				points.emplace_back( rotated, vector2 { x_offset, y_offset } );
			}
		}

		return points;
	}
}