#ifndef CRYPTOPP_CPP
#define CRYPTOPP_CPP

#include <impl/includes.hpp>

auto crypto::c_crypto::encrypt( const std::string& plaintext, const std::string& key, std::string& iv_out ) -> std::string
{
    if ( key.length( ) > CryptoPP::AES::MAX_KEYLENGTH )
    {
        log( HASH_STR( "encrypt failed because the private key is longer than %d symbols" ), CryptoPP::AES::MAX_KEYLENGTH );
        return HASH_STR( "" );
    }

    CryptoPP::AutoSeededRandomPool rng;
    CryptoPP::SecByteBlock iv( CryptoPP::AES::BLOCKSIZE );
    rng.GenerateBlock( iv, iv.size( ) );

    std::string ciphertext;

    try {
        CryptoPP::CBC_Mode<CryptoPP::AES>::Encryption encryptor;
        encryptor.SetKeyWithIV( reinterpret_cast< const BYTE* >( key.data( ) ), key.size( ), iv );

        CryptoPP::StringSource( plaintext, true,
            new CryptoPP::StreamTransformationFilter( encryptor,
                new CryptoPP::StringSink( ciphertext )
            )
        );

        std::string encoded_iv;
        CryptoPP::StringSource( iv, iv.size( ), true,
            new CryptoPP::Base64Encoder(
                new CryptoPP::StringSink( encoded_iv ), false // No newline
            )
        );

        std::string encoded_ciphertext;
        CryptoPP::StringSource( ciphertext, true,
            new CryptoPP::Base64Encoder(
                new CryptoPP::StringSink( encoded_ciphertext ), false // No newline
            )
        );

        iv_out = encoded_iv;
        return encoded_ciphertext;
    }
    catch ( const std::exception& e ) {
        log( HASH_STR( "encrypt failed because: %s" ), e.what( ) );
        return HASH_STR( "" );
    }
}

auto crypto::c_crypto::decrypt( const std::string& ciphertext, const std::string& key, const std::string& iv ) -> std::string
{
    if ( key.length( ) > CryptoPP::AES::MAX_KEYLENGTH )
    {
        log( HASH_STR( "decrypt failed because the private key is longer than %d symbols" ), CryptoPP::AES::MAX_KEYLENGTH );
        return HASH_STR( "" );
    }

    std::string decoded_iv;
    CryptoPP::StringSource( iv, true,
        new CryptoPP::Base64Decoder(
            new CryptoPP::StringSink( decoded_iv )
        )
    );

    std::string decoded_ciphertext;
    CryptoPP::StringSource( ciphertext, true,
        new CryptoPP::Base64Decoder(
            new CryptoPP::StringSink( decoded_ciphertext )
        )
    );

    std::string plaintext;

    try {
        CryptoPP::CBC_Mode<CryptoPP::AES>::Decryption decryptor;
        decryptor.SetKeyWithIV( reinterpret_cast< const BYTE* >( key.data( ) ), key.size( ), reinterpret_cast< const BYTE* >( decoded_iv.data( ) ) );

        CryptoPP::StringSource( decoded_ciphertext, true,
            new CryptoPP::StreamTransformationFilter( decryptor,
                new CryptoPP::StringSink( plaintext )
            )
        );

        return plaintext;
    }
    catch ( const std::exception& e ) {
        log( HASH_STR( "decrypt failed because: %s" ), e.what( ) );
        return HASH_STR( "" );
    }

}

auto crypto::c_crypto::base64_encode( const std::string& plaintext ) -> std::string
{
    BIO* bio, * b64;
    BUF_MEM* bufferPtr;

    b64 = BIO_new( BIO_f_base64( ) );
    bio = BIO_new( BIO_s_mem( ) );
    bio = BIO_push( b64, bio );

    BIO_set_flags( bio, BIO_FLAGS_BASE64_NO_NL );
    BIO_write( bio, plaintext.data( ), plaintext.size( ) );
    BIO_flush( bio );
    BIO_get_mem_ptr( bio, &bufferPtr );

    std::string encoded( bufferPtr->data, bufferPtr->length );
    BIO_free_all( bio );

    return encoded;
}

auto crypto::c_crypto::base64_decode( const std::string& ciphertext ) -> std::string
{
    BIO* bio, * b64;
    int decodeLen = ( int ) ciphertext.size( );
    std::vector<char> buffer( decodeLen );

    b64 = BIO_new( BIO_f_base64( ) );
    bio = BIO_new_mem_buf( ciphertext.data( ), ciphertext.size( ) );
    bio = BIO_push( b64, bio );

    BIO_set_flags( bio, BIO_FLAGS_BASE64_NO_NL );
    int length = BIO_read( bio, buffer.data( ), decodeLen );
    BIO_free_all( bio );

    return std::string( buffer.data( ), length );
}

#endif // !CRYPTOPP_CPP
