#ifndef UTILITY_CPP
#define UTILITY_CPP

#include <impl/includes.hpp>

auto utility::calculate_angle( const math::vector3& player, const math::vector3& target ) -> math::vector2 
{
	math::vector2 T;

	const auto direction = player - target;
	if ( !direction.valid( ) )
		return T;

	const auto magnitude = direction.magnitude( );

	const auto pitch = std::asinf( std::clamp( direction.y / magnitude, -1.0f, 1.0f ) ) * 57.29577951308232f;
	const auto yaw = -std::atan2f( direction.x, -direction.z ) * 57.29577951308232f;

	return math::vector2( pitch, yaw );
}

auto utility::get_cursor( )-> math::vector2 {

	POINT CursorPoint = {};
	GetCursorPos( &CursorPoint );

	return math::vector2( double( CursorPoint.x ), double( CursorPoint.y ) );
}

auto utility::is_hovered( math::vector2 start, math::vector2 size ) -> bool {

	auto cursor_pos = get_cursor( );

	if ( cursor_pos.x > start.x && cursor_pos.y > start.y ) {
		if ( cursor_pos.x < start.x + size.x && cursor_pos.y < start.y + size.y ) {
			return true;
		}
	}

	return false;
}

auto utility::handle_drag( ImVec2& draw_position, const ImVec2& top_left_rect, const ImVec2& rect_size, const bool& menu, drag_state& drag_state ) -> void {

	const auto mouse_position = get_cursor( );

	const bool left_mouse_down = ( GetAsyncKeyState( VK_LBUTTON ) & 0x8000 );
	const auto hovered = is_hovered( math::vector2( top_left_rect.x, top_left_rect.y ), math::vector2( rect_size.x, rect_size.y ) );

	if ( hovered && left_mouse_down && !drag_state.is_dragging ) {
		drag_state.drag_offset = ImVec2( mouse_position.x - draw_position.x, mouse_position.y - draw_position.y );
		drag_state.is_dragging = true;
	}

	if ( drag_state.is_dragging ) {
		if ( left_mouse_down && menu ) {
			draw_position.x = mouse_position.x - drag_state.drag_offset.x;
			draw_position.y = mouse_position.y - drag_state.drag_offset.y;
		}
		else {
			drag_state.is_dragging = false;
		}
	}
}

auto utility::random_float( float a, float b ) -> float {

	float random = ( ( float ) rand( ) ) / ( float ) RAND_MAX;
	float diff = b - a;
	float r = random * diff;
	return a + r;
}

auto utility::format_time( const float& time_in_minutes ) -> std::string {

	int days = static_cast< int >( time_in_minutes ) / ( 24 * 60 );
	int remaining_minutes = static_cast< int >( time_in_minutes ) % ( 24 * 60 );
	int hours = remaining_minutes / 60;
	int minutes = hours % 60;

	std::ostringstream formatted {};

	formatted << days << "d : " << hours << "h : " << minutes << "m";

	return formatted.str( );
}

auto utility::lerp( float default_val, float target, float alpha ) -> float {

	return default_val + alpha * ( target - default_val );
}

auto utility::line_cirlce_intersection( math::vector3 center, float radius, math::vector3 raystart, math::vector3 rayend ) -> bool {

	math::vector2 p( raystart.x, raystart.z );
	math::vector2 q( rayend.x, rayend.z );

	float a = q.y - p.y;
	float b = p.x - q.x;
	float c = ( a * ( p.x ) + b * ( p.y ) ) * -1.f;

	float x = center.x;
	float y = center.z;

	float c_x = ( b * ( ( b * x ) - ( a * y ) ) - a * c ) / ( std::pow( a, 2 ) + std::pow( b, 2 ) );
	float c_y = ( a * ( ( -b * x ) + ( a * y ) ) - ( b * c ) ) / ( std::pow( a, 2 ) + std::pow( b, 2 ) );

	math::vector2 closestpoint( c_x, c_y );

	float distance = p.distance( q );

	if ( p.distance( closestpoint ) > distance || q.distance( closestpoint ) > distance )
		return false;

	if ( radius > closestpoint.distance( math::vector2( center.x, center.z ) ) )
	{
		math::vector2 p( raystart.x, raystart.y );
		math::vector2 q( rayend.x, rayend.y );

		float a = q.y - p.y;
		float b = p.x - q.x;
		float c = ( a * ( p.x ) + b * ( p.y ) ) * -1.f;

		float x = center.x;
		float y = center.y;

		float c_x = ( b * ( ( b * x ) - ( a * y ) ) - a * c ) / ( std::pow( a, 2 ) + std::pow( b, 2 ) );
		float c_y = ( a * ( ( -b * x ) + ( a * y ) ) - ( b * c ) ) / ( std::pow( a, 2 ) + std::pow( b, 2 ) );

		math::vector2 closestpoint( c_x, c_y );
		if ( radius > closestpoint.distance( math::vector2( center.x, center.y ) ) ) return true;
		else return false;
	}

	return false;
}

auto utility::get_lerp( math::vector3 center, math::vector3 firstposition, math::vector3 lastposition ) -> float {
	
	math::vector3 center2d = center; math::vector3 first2d = firstposition; math::vector3 last2d = lastposition;
	center2d.y = 0.f; first2d.y = 0.f; last2d.y = 0.f;

	float distancefromfirst = ( center2d - first2d ).magnitude( );
	float distancefromlast = ( center2d - last2d ).magnitude( );

	float totaldistance = ( last2d - first2d ).magnitude( );

	if ( totaldistance == 0 )
		return 0.5;
	else
	{
		float percentfromfirst = distancefromfirst / totaldistance;
		float percentfromlast = distancefromlast / totaldistance;

		float lerpvalue = percentfromfirst / ( percentfromfirst + percentfromlast );
		return lerpvalue;
	}
}

auto utility::cos_tan_sin_line_h( float angle, float range, std::int32_t x, std::int32_t y, std::int32_t length ) -> math::vector2 {
	
	const auto yaw = ( angle + 45.00f ) * ( 3.1415926535f / 180.00f );

	const auto view_cos = cos( yaw );
	const auto view_sin = sin( yaw );

	const auto x2 = range * ( -view_cos ) + range * view_sin;
	const auto y2 = range * ( -view_cos ) - range * view_sin;

	return { ( float ) ( x + ( std::int32_t ) ( x2 / range * ( length ) ) ), ( float ) ( y + ( std::int32_t ) ( y2 / range * ( length ) ) ) };
}

auto utility::calculate_radar_point( math::vector3 position, math::vector3 local, float distance, float angle, float size, float range ) -> math::vector2 {

	const auto y = local.x - position.x;
	const auto x = local.z - position.z;

	const auto num = std::atan2( y, x ) * 57.29578f - 270.00f - angle;

	return { ( distance * std::cos( num * 0.0174532924f ) ) * ( size / range ) / 2.00f, ( distance * std::sin( num * 0.0174532924f ) ) * ( size / range ) / 2.00f };
}

auto utility::update_function( const std::uint32_t& time, std::chrono::steady_clock::time_point& timestamp, const std::function<void( )>& task ) -> void {

	auto current_time = std::chrono::steady_clock::now( );
	if ( ( current_time - timestamp ) >= std::chrono::milliseconds( time ) ) {
		task( );
		timestamp = current_time;
	}
}

auto utility::select_file( const std::wstring& extension, const std::wstring& description, const std::wstring& pattern ) -> std::wstring {

	wchar_t filename [ MAX_PATH ] = L"";

	OPENFILENAMEW ofn;
	ZeroMemory( &ofn, sizeof( ofn ) );
	ofn.lStructSize = sizeof( ofn );
	ofn.hwndOwner = nullptr;

	std::wstring filter = description + L'\0' + pattern + L'\0';

	ofn.lpstrFilter = filter.c_str( );
	ofn.lpstrFile = filename;
	ofn.nMaxFile = MAX_PATH;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
	ofn.lpstrDefExt = extension.c_str( );

	if ( GetOpenFileNameW( &ofn ) )
	{
		SetWindowPos( GetParent( ofn.hwndOwner ), HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE );
		return filename;
	}

	return L"";
}

auto utility::select_folder( const std::wstring& title ) -> fs::path {

	BROWSEINFOW bi = { 0 };
	wchar_t display_name [ MAX_PATH ] = { 0 };

	bi.hwndOwner = NULL;
	bi.lpszTitle = title.c_str( );
	bi.ulFlags = BIF_RETURNONLYFSDIRS | BIF_NEWDIALOGSTYLE;
	bi.pszDisplayName = display_name;

	LPITEMIDLIST pidl = SHBrowseForFolderW( &bi );
	if ( pidl ) {
		wchar_t selected_path [ MAX_PATH ] = { 0 };
		if ( SHGetPathFromIDListW( pidl, selected_path ) ) {
			CoTaskMemFree( pidl );
			return fs::path( selected_path );
		}
		CoTaskMemFree( pidl );
	}

	return fs::path( );
}

void utility::copy_to_clipboard( const std::string& string )
{
	LI_FN( OpenClipboard ).safe_cached( )( 0 );
	LI_FN( EmptyClipboard ).safe_cached( )( );

	HGLOBAL hg = GlobalAlloc( GMEM_MOVEABLE, string.size( ) );
	if ( !hg ) {
		LI_FN( CloseClipboard ).safe_cached( )( );
		return;
	}

	memcpy( LI_FN( GlobalLock ).safe_cached( )( hg ), string.c_str( ), string.size( ) );

	LI_FN( GlobalUnlock ).safe_cached( )( hg );

	LI_FN( SetClipboardData ).safe_cached( )( CF_TEXT, hg );
	LI_FN( CloseClipboard ).safe_cached( )( );

	LI_FN( GlobalFree ).safe_cached( )( hg );
}

#endif // !UTILITY_CPP
