#ifndef EFI_HPP
#define EFI_HPP

constexpr ULONG PAYLOAD_CONTROL_CODE = CTL_CODE( FILE_DEVICE_UNKNOWN, 0x282, METHOD_BUFFERED, FILE_SPECIAL_ACCESS );

constexpr auto min_user_addr = 0x0000000000010000;
constexpr auto max_user_addr = 0x00007FFFFFFEFFFF;

namespace memory
{
	__forceinline uintptr_t to_addr( const void* pointer )
	{
		return reinterpret_cast< uintptr_t >( pointer );
	}

	__forceinline void* to_ptr( uintptr_t address )
	{
		return reinterpret_cast< void* >( address );
	}

	__forceinline bool is_valid( uintptr_t address )
	{
		return ( address >= min_user_addr && address < max_user_addr );
	}

	__forceinline bool is_valid( const void* pointer )
	{
		return is_valid( to_addr( pointer ) );
	}
}

namespace efi_hook
{
	enum class setup_error_code
	{
		pid_unavailable,
		payload_communication_failure,
		payload_mapping_failure,
		dtb_unavailable,
		system_operational,
		assembly_error,
		open_library_error,
		unity_error
	};

	enum class game_error_code
	{
		empty_process_name,
		system_operational
	};

	typedef enum _requests
	{
		invoke_start,
		invoke_base,
		invoke_read,
		invoke_is_mapped,
		invoke_hide_window,
		invoke_hide_process,
		invoke_write,
		invoke_resolve_dtb,
		invoke_query_memory,
		invoke_peb,
		invoke_module_base,
		invoke_success,
		invoke_unique,
		invoke_unload,
	}requests, * prequests;

	typedef struct _read_invoke
	{
		uint32_t pid;
		bool multipage;
		uintptr_t address;
		void* buffer;
		size_t size;
	} read_invoke, * pread_invoke;

	typedef struct _write_invoke
	{
		uint32_t pid;
		bool multipage;
		uintptr_t address;
		void* buffer;
		size_t size;
	} write_invoke, * pwrite_invoke;

	typedef struct _is_mapped_invoke
	{
		bool is_mapped;
	} is_mapped_invoke, * pis_mapped;

	typedef struct _hide_attributes_invoke
	{
		uintptr_t handle;
		bool status;
	} hide_attributes_invoke, * phide_attributes_invoke;

	typedef struct _query_memory_invoke
	{
		uint32_t pid;
		uintptr_t address;
		uintptr_t page_base;
		uint32_t page_prot;
		size_t page_size;
		bool status;
	} query_memory_invoke, * pquery_memory_invoke;

	typedef struct _hide_process_invoke
	{
		const char* name;
		bool status;
	} hide_process_invoke, * phide_process_invoke;

	typedef struct _dtb_invoke
	{
		uint32_t pid;
		uintptr_t handle;
	} dtb_invoke, * pdtb_invoke;

	typedef struct _base_invoke
	{
		uint32_t pid;
		uintptr_t handle;
	} base_invoke, * pbase_invoke;

	typedef struct _peb_invoke
	{
		uint32_t pid;
		ULONGLONG handle;
	} peb_invoke, * ppeb_invoke;

	typedef struct _invoke_data
	{
		uint32_t unique;
		requests code;
		void* data;
	}invoke_data, * pinvoke_data;

	typedef struct _request_data
	{
		uintptr_t key;
	}request_data, * prequest_data;

	class efi_t {
	public:
		std::int32_t process_id;
		std::uint64_t image_base;
		std::uint64_t ntos_base;
		std::uint64_t cache_rax;
		std::uint64_t response_buffer;
		std::uint64_t m_gameassembly;
		std::uint64_t unity_player;

		HANDLE payload_handle;

	private:
		std::uint64_t cr3_header;
	public:
		uint32_t efi_read_large_array_count = 0;
		std::uint64_t text_section;

		[[nodiscard]] bool send_payload_cmd( void* data, requests parameters );
		[[nodiscard]] bool is_payload_available( );

		[[nodiscard]] setup_error_code setup( const std::wstring& proc_name );
		[[nodiscard]] game_error_code wait_for_proccess( const std::wstring& process_name );

		const std::uintptr_t get_image_ps_base( );

		const std::uintptr_t get_image_base( const std::uintptr_t e_process );
		const std::uint32_t get_process_pid( const std::wstring& proc_name );

		bool read_large_array( std::uintptr_t address, void* buffer, std::size_t size );

		// magic stuff.
		// call only once.
		std::uintptr_t patch_directory_table_base( );
		std::uintptr_t get_peb( );

		bool hide_window_attributes( std::uintptr_t window_handle );

		// vm

		std::uintptr_t get_module_base( const wchar_t* module_name );
		bool read_physical( const uintptr_t address, void* buffer, const std::uintptr_t size, bool multipage );
		bool write_physical( const uintptr_t address, void* buffer, const size_t size );

		// vm templates

		template <typename t>
		auto write( const uintptr_t address, t value ) -> bool {
			return write_physical( address, &value, sizeof( t ) );
		}

		template <class t>
		auto read( const uintptr_t address, bool multipage = false ) -> t {
			t response { };

			auto result = read_physical(
				address,
				&response,
				sizeof( t ),
				multipage
			);

			return response;
		}

		inline uint64_t pattern_scan( uint64_t module, const char* signature, const char* sectionName = nullptr, int skip = 0 )
		{
			static auto pattern_to_byte = [ ] ( const char* pattern ) {
				auto bytes = std::vector<int> {};
				auto start = const_cast< char* >( pattern );
				auto end = const_cast< char* >( pattern ) + strlen( pattern );

				for ( auto current = start; current < end; ++current ) {
					if ( *current == '?' ) {
						++current;
						if ( *current == '?' )
							++current;
						bytes.push_back( -1 );
					}
					else {
						bytes.push_back( strtoul( current, &current, 16 ) );
					}
				}
				return bytes;
				};

			auto dosHeader = ( PIMAGE_DOS_HEADER ) module;
			auto ntHeaders = ( PIMAGE_NT_HEADERS ) ( ( std::uint8_t* ) module + dosHeader->e_lfanew );
			auto patternBytes = pattern_to_byte( signature );
			auto s = patternBytes.size( );
			auto d = patternBytes.data( );
			int currentskip = 0;

			if ( !sectionName ) {
				auto sizeOfImage = ntHeaders->OptionalHeader.SizeOfImage;
				auto scanBytes = reinterpret_cast< std::uint8_t* >( module );

				for ( auto i = 0ul; i < sizeOfImage - s; ++i ) {
					if ( currentskip < skip ) {
						currentskip++;
						continue;
					}

					bool found = true;
					for ( auto j = 0ul; j < s; ++j ) {
						if ( scanBytes [ i + j ] != d [ j ] && d [ j ] != -1 ) {
							found = false;
							break;
						}
					}
					if ( found ) {
						return ( uintptr_t ) &scanBytes [ i ];
					}
				}
			}
			else {
				auto sectionHeader = IMAGE_FIRST_SECTION( ntHeaders );
				for ( WORD i = 0; i < ntHeaders->FileHeader.NumberOfSections; ++i, ++sectionHeader ) {
					if ( strncmp( reinterpret_cast< const char* >( sectionHeader->Name ), sectionName, IMAGE_SIZEOF_SHORT_NAME ) == 0 ) {
						auto sectionStart = reinterpret_cast< std::uint8_t* >( module ) + sectionHeader->VirtualAddress;
						auto sectionSize = sectionHeader->Misc.VirtualSize;

						for ( auto j = 0ul; j < sectionSize - s; ++j ) {
							bool found = true;
							for ( auto k = 0ul; k < s; ++k ) {
								if ( sectionStart [ j + k ] != d [ k ] && d [ k ] != -1 ) {
									found = false;
									break;
								}
							}
							if ( found ) {
								if ( currentskip < skip ) {
									currentskip++;
									continue;
								}
								return ( uintptr_t ) &sectionStart [ j ];
							}
						}
						break;
					}
				}
			}
			return ( uintptr_t )nullptr;
		}

		auto rva_to_va( uintptr_t scan_result, int offset ) -> std::uintptr_t
		{
			uintptr_t instruction_address = scan_result;
			uintptr_t next_instruction_address = instruction_address + offset + 4;

			return next_instruction_address + read<int>( instruction_address + offset );
		}

		// payload
		[[nodiscard]] bool create_payload_handle( );

		[[nodiscard]] auto is_valid( uintptr_t pointer ) -> bool {
			return ( pointer && pointer > 0xFFFFFF && pointer < 0x7FFFFFFFFFFF && pointer != NULL );
		}

		template<typename t>
		auto read_chain( const std::uint64_t& address, std::vector<std::uint64_t> offsets ) -> t
		{
			if ( !this->is_valid( address ) ) return {};

			auto value = address;

			for ( auto idx = 0; idx < offsets.size( ) - 1; idx++ )
			{
				if ( !this->is_valid( value ) ) return {};

				const auto offset = offsets [ idx ];

				value = this->read<std::uint64_t>( value + offset );
			}

			if ( !this->is_valid( value ) ) return { };

			return this->read<t>( value + offsets [ offsets.size( ) - 1 ] );
		};

		auto read_string( std::uintptr_t address ) -> std::string
		{
			if ( !this->is_valid( address ) )
				return HASH_STR( "Undefined" );

			char buffer [ 128 ] = { 0 };

			this->read_large_array( address, buffer, sizeof( buffer ) - 1 );
			buffer [ sizeof( buffer ) - 1 ] = '\0';

			return std::string( buffer );
		}

		template <typename T>
		auto read_vec( std::uintptr_t base, std::uint32_t count ) -> std::vector<T>
		{
			std::vector<T> result = { };

			for ( int index = 0; index < count; index++ ) 
			{
				T entry = read<T>( base + sizeof( T ) * index, true );
				result.push_back( entry );
			}

			return result;
		}
	};
} inline auto g_vm = std::make_shared<efi_hook::efi_t>( );


#endif